/* GNOME snmp load panel applet
 * (C) 2000 Frank Strauss <strauss@escape.de>
 *
 * Author: Frank Strauss <strauss@escape.de>
 * With code from netload_applet
 * With code extensively stolen from: Tim P. Gerla
 * With code from wmload.c, v0.9.2, apparently by Ryan Land, rland@bc1.com.
 *
 */

#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>
#include <time.h>
#include <netinet/in.h>
#include <netdb.h>
#include <signal.h>

#include <config.h>

#include <gnome.h>
#include <gdk/gdkx.h>
#include <applet-widget.h>

#include <ucd-snmp/ucd-snmp-config.h>
#include <ucd-snmp/default_store.h>
#include <ucd-snmp/asn1.h>
#include <ucd-snmp/snmp.h>
#include <ucd-snmp/snmp_api.h>
#include <ucd-snmp/snmp_client.h>
#include <ucd-snmp/mib.h>

#include "properties.h"



void start_timer(void);

GtkWidget *snmpload;
GdkPixmap *pixmap;
GtkWidget *disp;
GdkGC *gc, *gc1, *gc2;
GdkColor color1, color2, colorb, colora, colorl;
snmpload_properties props;
int timer_index = -1;
struct snmp_session *ss = NULL;
struct snmp_pdu *pdu;
int linestatus = 0, tooltip_set = 0;
GtkWidget *applet;



oid *snmp_parse_oid(char *argv,
		    oid *root,
		    size_t *rootlen)
{
    if (read_objid(argv, root, rootlen)) {
	return root;
    }
    if (get_node(argv, root, rootlen)) {
	return root;
    }
    return NULL;
}



/*
 * Use a circular buffer for storing the values.
 */

static int draw(void)
{
    static gint width = 0;
    static unsigned long int *data1 = NULL;
    static unsigned long int *data2 = NULL;
    static unsigned long int old_bytes1, old_bytes2, max_delta;
    static int max_delta_pos = -1;
    unsigned long int bytes1 = 0, bytes2 = 0;
    struct variable_list *vars;
    static int init = 0;
    static int front = 0;
    int	i, p;
    static float scale_factor = 0;
    struct snmp_pdu *response;
    int status;
    oid statusname[MAX_OID_LEN], v1name[MAX_OID_LEN], v2name[MAX_OID_LEN];
    size_t statuslength, v1length, v2length;
    int height = disp->allocation.height;
    static char linetime[30];
    time_t t;
    
    if (width != disp->allocation.width || data1 == NULL) {
	/* Resize horizontally. */
	width = disp->allocation.width;
#if 1
	if (data1) {
	    free(data1);
	    free(data2);
	}
	data1 = (unsigned long int *)calloc(width, sizeof(unsigned long int));
	data2 = (unsigned long int *)calloc(width, sizeof(unsigned long int));
	max_delta_pos = -1;
	max_delta = 0;
#else
	data1 = (unsigned long int *)
	    realloc(data1, width * sizeof(unsigned long int));
	data2 = (unsigned long int *)
	    realloc(data2, width * sizeof(unsigned long int));
#endif
    }

    if (ss) {
	pdu = snmp_pdu_create(SNMP_MSG_GET);

	statuslength = MAX_OID_LEN;
	if (snmp_parse_oid(props.oidstatus, statusname, &statuslength)) {
	    snmp_add_null_var(pdu, statusname, statuslength);
	}
    
	v1length = MAX_OID_LEN;
	if (snmp_parse_oid(props.oidv1, v1name, &v1length)) {
	    snmp_add_null_var(pdu, v1name, v1length);
	}
	
	v2length = MAX_OID_LEN;
	if (snmp_parse_oid(props.oidv2, v2name, &v2length)) {
	    snmp_add_null_var(pdu, v2name, v2length);
	}
    
	status = snmp_synch_response(ss, pdu, &response);

	if (status == STAT_SUCCESS) {
	    if (response->errstat == SNMP_ERR_NOERROR) {
		for (vars = response->variables; vars;
		     vars = vars->next_variable) {
		    /* print_variable(vars->name, vars->name_length, vars); */
		    if (!snmp_oid_compare(vars->name, vars->name_length,
					  statusname, statuslength)) {
			linestatus = *(unsigned long *)(vars->val.integer);
		    }
		    if (!snmp_oid_compare(vars->name, vars->name_length,
					  v1name, v1length)) {
			bytes1 = *(unsigned long *)(vars->val.integer);
		    }
		    if (!snmp_oid_compare(vars->name, vars->name_length,
					  v2name, v2length)) {
			bytes2 = *(unsigned long *)(vars->val.integer);
		    }
		}
	    } else {
		/* error */
	    }
	} else if (status == STAT_TIMEOUT) {
	    /* timeout: no response */
	} else {
	    /* other error */
	}

	if (response)
	    snmp_free_pdu(response);
    }
    
    if (!init) {
	init = 1;
	old_bytes1 = bytes1;
	old_bytes2 = bytes2;
    }

    /* The MAX is in case the stats get reset or wrap.
     * It'll get one wrong sample.
     */
    data1[front] = MAX(0, ((signed int)(bytes1 - old_bytes1)) /
		       ((float)props.speed / 1000));
    data2[front] = MAX(0, ((signed int)(bytes2 - old_bytes2)) /
		       ((float)props.speed / 1000));

    if (front == max_delta_pos) {
	/* The maximum has scrolled off. Rescale. */
	max_delta = data1[0] + data2[0];
	max_delta_pos = 0;
	for (i = 0; i < width; i++) {
	    if ((data1[i] + data2[i]) > max_delta) {
		max_delta = data1[i] + data2[i];
		scale_factor = (float)height / (float)max_delta;
		max_delta_pos = i;
	    }
	}
    }

    if ((data1[front] + data2[front]) > max_delta) {
	max_delta = data1[front] + data2[front];
	max_delta_pos = front;
	scale_factor = (float)height / (float)max_delta;
    }

    /* Fill rectangle with colorb if offline or colora if online. */
    if (linestatus == props.up) {
	gdk_gc_set_foreground(gc, &colora);
	if (tooltip_set == 0) {
	    t = time(NULL);
	    sprintf(linetime, _("Online since %s"), ctime(&t));
	    linetime[strlen(linetime)-1] = '.';
	    applet_widget_set_tooltip(APPLET_WIDGET(applet), linetime);
	    tooltip_set = 1;
	}
    } else {
	gdk_gc_set_foreground(gc, &colorb);
	if (tooltip_set == 1) {
	    t = time(NULL);
	    sprintf(linetime, _("Offline since %s"), ctime(&t));
	    linetime[strlen(linetime)-1] = '.';
	    applet_widget_set_tooltip(APPLET_WIDGET(applet), linetime);
	    tooltip_set = 0;
	}
    }
    gdk_draw_rectangle(pixmap, gc,
		       TRUE, 0, 0,
		       disp->allocation.width,
		       disp->allocation.height);
    
    p = width;
    gdk_gc_set_foreground(gc1, &color1);
    gdk_gc_set_foreground(gc2, &color2);
    for(i = front; i >= 0; i--) {
	if (data1[i]) {
	    gdk_draw_line(pixmap, gc1, p, height,
			  p, (int)(height - (data1[i] * scale_factor)));
	}
	if (data2[i]) {
	    gdk_draw_line(pixmap, gc2,
			  p, (int)(height - (data1[i] * scale_factor)),
			  p, (int)(height - ((data1[i] + data2[i]) * scale_factor)));
	}
	p--;
    }

    for (i = width - 1; i > front; i--) {
	if (data1[i]) {
	    gdk_draw_line(pixmap, gc1, p, height,
			  p, (int)(height - (data1[i] * scale_factor)));
	}
	if (data2[i]) {
	    gdk_draw_line(pixmap, gc2,
			  p, (int)(height - (data1[i] * scale_factor)),
			  p, (int)(height - ((data1[i] + data2[i]) * scale_factor)));
	}
	p--;
    }

    front++;
    if (front >= width)
	front = 0;

    old_bytes1 = bytes1;
    old_bytes2 = bytes2;

    /* Draw horizontal lines if appropriate. */
    gdk_gc_set_foreground(gc, &colorl);
    for (i = props.line; props.line && i < max_delta; i += props.line) {
	gdk_draw_line(pixmap, gc,
		      0, height - (int)(i * scale_factor),
		      width, height - (int)(i * scale_factor));
    }
		
    gdk_draw_pixmap(disp->window,
		    disp->style->fg_gc[GTK_WIDGET_STATE(disp)],
		    pixmap,
		    0, 0,
		    0, 0,
		    disp->allocation.width,
		    disp->allocation.height);

    return TRUE;
}



gint snmpload_configure(GtkWidget *widget, GdkEventConfigure *event)
{
    pixmap = gdk_pixmap_new(widget->window,
			    widget->allocation.width,
			    widget->allocation.height,
			    gtk_widget_get_visual(disp)->depth);
    gdk_draw_rectangle(pixmap,
		       widget->style->black_gc,
		       TRUE, 0,0,
		       widget->allocation.width,
		       widget->allocation.height);
    gdk_draw_pixmap(widget->window,
		    disp->style->fg_gc[GTK_WIDGET_STATE(widget)],
		    pixmap,
		    0, 0,
		    0, 0,
		    disp->allocation.width,
		    disp->allocation.height);
    return TRUE;
} 



static gint clicked_cb(GtkWidget *widget, GdkEventButton *e, gpointer data)
{
    char *argv[4];
	
    if (e->button != 1) {
	/* Ignore buttons 2 and 3 */
	return FALSE; 
    }

    if (fork() == 0) {
	argv[0] = "sh";
	argv[1] = "-c";
	if (linestatus == props.up) {
	    argv[2] = props.commanddown;
	} else {
	    argv[2] = props.commandup;
	}
	argv[3] = 0;
	execve("/bin/sh", argv, NULL);
	exit(0);
    }
    
    return TRUE; 
}



static gint snmpload_expose(GtkWidget *widget, GdkEventExpose *event)
{
    gdk_draw_pixmap(widget->window,
		    widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
		    pixmap,
		    event->area.x, event->area.y,
		    event->area.x, event->area.y,
		    event->area.width, event->area.height);
    return FALSE;
}



static GtkWidget *snmpload_new(void)
{
    GtkWidget *frame, *box;
    GtkWidget *event_box;
    
    box = gtk_vbox_new(FALSE, FALSE);
    gtk_widget_show(box);

    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame),
			      props.look ? GTK_SHADOW_OUT : GTK_SHADOW_IN);

    disp = gtk_drawing_area_new();
    gtk_signal_connect(GTK_OBJECT(disp), "expose_event",
		       (GtkSignalFunc)snmpload_expose, NULL);
    gtk_signal_connect(GTK_OBJECT(disp), "configure_event",
		       (GtkSignalFunc)snmpload_configure, NULL);
    gtk_widget_set_events(disp, GDK_EXPOSURE_MASK);
	
    gtk_box_pack_start_defaults(GTK_BOX(box), disp);
    // gtk_container_add(GTK_CONTAINER(frame), box);

    event_box = gtk_event_box_new();
    gtk_widget_show(event_box);
    gtk_widget_set_events(event_box, GDK_BUTTON_PRESS_MASK);
    gtk_signal_connect(GTK_OBJECT(event_box), "button_press_event",
		       GTK_SIGNAL_FUNC(clicked_cb), NULL);

    gtk_container_add(GTK_CONTAINER(event_box), box);
    gtk_container_add(GTK_CONTAINER(frame), event_box);

    gtk_widget_set_usize(disp, props.width, props.height);

    start_timer();
        
    gtk_widget_show_all(frame);
    return frame;
}



void start_timer(void)
{
    if (timer_index != -1)
	gtk_timeout_remove(timer_index);

    timer_index = gtk_timeout_add(props.speed, (GtkFunction)draw, NULL);
}



void setup_colors(void)
{
    GdkColormap *colormap;

    colormap = gtk_widget_get_colormap(disp);
                
    gdk_color_parse(props.color1, &color1);
    gdk_color_alloc(colormap, &color1);

    gdk_color_parse(props.color2, &color2);
    gdk_color_alloc(colormap, &color2);

    gdk_color_parse(props.colorb, &colorb);
    gdk_color_alloc(colormap, &colorb);

    gdk_color_parse(props.colora, &colora);
    gdk_color_alloc(colormap, &colora);

    gdk_color_parse(props.colorl, &colorl);
    gdk_color_alloc(colormap, &colorl);
}
	        


void setup_size(void)
{
    gtk_widget_set_usize(disp, props.width, props.height);
}



void setup_snmp(void)
{
    struct snmp_session ss_init;
    
    if (ss)
	snmp_close(ss);
    
    snmp_sess_init(&ss_init);
    ss_init.version = SNMP_VERSION_1;
    ss_init.peername = props.agent;
    ss_init.remote_port = props.port;
    ss_init.community = props.community;
    ss_init.community_len = strlen(ss_init.community);
    ss_init.retries = props.retries;
    ss_init.timeout = props.timeout * 1000;
    
    ss = snmp_open(&ss_init);
    
    if (ss == NULL) {
	snmp_sess_perror("snmpload_applet", &ss_init);
	exit(1);
    }
}



static void create_gc(void)
{
        gc = gdk_gc_new(disp->window);
        gc1 = gdk_gc_new(disp->window);
        gc2 = gdk_gc_new(disp->window);
        gdk_gc_copy(gc, disp->style->white_gc);
}



static gint applet_save_session(GtkWidget *widget, char *privcfgpath,
				char *globcfgpath, gpointer data)
{
	save_properties(privcfgpath, &props);
	return FALSE;
}



static void
error_close_cb(GtkWidget *widget, void *data)
{
  applet_widget_remove(APPLET_WIDGET(widget));

  if (ss)
      snmp_close(ss);
}



/*
 * An error occured.
 */
void
error_dialog(char *message, int fatal)
{
	static GtkWidget *error = NULL;
	GtkWidget *less, *label;

	if (error) {
	    return;
	}

	error = gnome_dialog_new(_("SNMPload Error"),
				 GNOME_STOCK_BUTTON_CLOSE, NULL);
	gnome_dialog_set_close(GNOME_DIALOG(error), TRUE);
	gnome_dialog_close_hides(GNOME_DIALOG(error), TRUE);

	less = gnome_less_new();
	label = gtk_label_new(_("An error occured in the SNMPload Applet:"));
	
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(error)->vbox), label,
			   FALSE, FALSE, GNOME_PAD);
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(error)->vbox), less,
			   TRUE, TRUE, GNOME_PAD);

	gnome_less_fixed_font(GNOME_LESS(less));
	gtk_window_set_modal(GTK_WINDOW(error), TRUE);

	if (fatal)
	    gtk_signal_connect(GTK_OBJECT(error),
			       "clicked",
			       GTK_SIGNAL_FUNC(error_close_cb), NULL);

	gtk_widget_show(less);
	gtk_widget_show(label);
	gtk_widget_show(error);
	gnome_less_show_string(GNOME_LESS(less), message);
}



static void
about_cb(AppletWidget *widget, gpointer data)
{
	GtkWidget *about;
	static const gchar *authors[] = {
	    "Frank Strau (strauss@escape.de)",
	    NULL
	};

	about = gnome_about_new(
	    _("The GNOME SNMP Counter Load Applet"), VERSION,
	    "(C) 2000 Frank Strau",
	    authors,
	    _("This applet is released under the terms and conditions "
	      "of the GNU Public Licence.\n"
	      "It displays one or two values of SNMP counter "
	      "variables retrieved from an SNMP agent, e.g. "
	      "ifInOctets and ifOutOctets of a network interface. "
	      "A third object is queried to display the interface status "
	      "using different background colors, e.g. ifOperStatus."),
	    NULL);
	gtk_widget_show(about);
	return;
}



int main(int argc, char **argv)
{
    /* Initialize the i18n stuff */
    bindtextdomain(PACKAGE, GNOMELOCALEDIR);
    textdomain(PACKAGE);

    applet_widget_init("snmpload_applet", VERSION, argc,
		       argv, NULL, 0, NULL);

    applet = applet_widget_new("snmpload_applet");
    if (!applet)
	g_error(_("Can't create applet!\n"));
    
    load_properties(APPLET_WIDGET(applet)->privcfgpath, &props);
	
    init_snmp("snmpload_applet");

    snmpload = snmpload_new();
    applet_widget_add(APPLET_WIDGET(applet), snmpload);
    gtk_widget_show(applet);

    applet_widget_set_tooltip(APPLET_WIDGET(applet),
			      _("No connection since started."));

    create_gc();
    setup_colors();
    setup_snmp();

    gtk_signal_connect(GTK_OBJECT(applet),"save_session",
		       GTK_SIGNAL_FUNC(applet_save_session),
		       NULL);
	
    applet_widget_register_stock_callback(APPLET_WIDGET(applet),
					  "about",
					  GNOME_STOCK_MENU_ABOUT,
					  _("About..."),
					  about_cb,
					  NULL);

    applet_widget_register_stock_callback(APPLET_WIDGET(applet),
					  "properties",
					  GNOME_STOCK_MENU_PROP,
					  _("Properties..."),
					  properties,
					  NULL);

    applet_widget_gtk_main();
    return 0;
}
