/*
 * Copyright (c) 2006, Swedish Institute of Computer Science.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
/*---------------------------------------------------------------------------*/
#define MY_CLOCK_SECOND CLOCK_SECOND * 4
#define MIN_CHANNEL 11
#define MAX_CHANNEL 26
#define CHANNEL_SCAN_TIME MY_CLOCK_SECOND
/*---------------------------------------------------------------------------*/
#include "contiki.h"

#include "dev/uart1.h"
#include "overloads/cc2420.h"

#include "dev/leds.h"

#include "dev/cc2420.h"
#include "dev/cc2420_const.h"

#include "sys/clock.h"

#include <stdio.h> /* For printf() */

#include "ginseng-slip.h"
#include "sniffer.h"
/*---------------------------------------------------------------------------*/
PROCESS(sniffer_process, "802.15.4 Packet Sniffer");
AUTOSTART_PROCESSES(&sniffer_process);

u16_t stats_packets_on_channel_crc;
u16_t stats_packets_on_channel_nocrc;
u8_t  stats_channel;

u8_t  status_channel_scan;
u8_t  status_sniffing;
u8_t  status_rssi;

u8_t  rssi_history[10];
u8_t  rssi_ptr;

static struct etimer ledTimer;
static struct etimer scanTimer;
static struct etimer startupTimer;
/*---------------------------------------------------------------------------*/
void radioReceiver(const struct radio_driver * d) {
  u8_t buffer[200];
  u8_t length;
  u8_t i;

  rtimer_clock_t start = rtimer_arch_now();

  length = cc2420_read(&buffer, 200);

  if( status_sniffing ) {
    // Packet Format:
    // 16 bit: rtimer_arch_now
    //  8 bit: packet length
    //         PACKET

    // Send the type
    slipSendChar(SLIP_TYPE_SNIFFED);

    // Then the timestamp
    slipSendChar((start & 0xFF00) >>  8);
    slipSendChar((start & 0x00FF) >>  0);

    // Send the channel that is currently in use
    slipSendChar(stats_channel);

    // Now the length
    slipSendChar(length & 0xFF);

    // And the raw packet data
    for(i=0; i<length; i++) {
      slipSendChar(buffer[i]);
    }

    uart1_writeb(SLIP_END);
  }
 
  leds_toggle(LEDS_GREEN);

  // Count packets with matching CRC differently
  if( buffer[length - 2] & 0x80 ) {
    stats_packets_on_channel_crc ++;
  } else {
    stats_packets_on_channel_nocrc ++;
  }
}
/*---------------------------------------------------------------------------*/
PROCESS_THREAD(sniffer_process, ev, data)
{
  PROCESS_BEGIN();

  serial_buffer_event = process_alloc_event();

  slipInit();

  status_channel_scan = 1;
  stats_packets_on_channel_crc = 0;
  stats_packets_on_channel_nocrc = 0;
  stats_channel = MIN_CHANNEL;
  status_sniffing = 1;
  status_rssi = 0;

  memset(rssi_history, 0, sizeof(rssi_history));
  rssi_ptr = 0;
 
  cc2420_init();  
  cc2420_set_receiver(radioReceiver);
  cc2420_on();

  cc2420_set_channel(stats_channel);

  etimer_set(&ledTimer, MY_CLOCK_SECOND);
  etimer_set(&startupTimer, MY_CLOCK_SECOND*2);

  // Give the PC a chance to abort the channelscan
  PROCESS_YIELD_UNTIL(etimer_expired(&startupTimer)); 

  if( status_channel_scan ) {
    slipSendShortMessage(SLIP_TYPE_CHANNEL_SCAN_START, 0);
    slipSendShortMessage(SLIP_TYPE_CHANNEL, stats_channel);
    etimer_set(&scanTimer, CHANNEL_SCAN_TIME);
  }

  leds_on(LEDS_BLUE); 

  while(1) {
    PROCESS_YIELD_UNTIL(etimer_expired(&ledTimer) || etimer_expired(&scanTimer) || ev == serial_buffer_event);

    if( etimer_expired(&ledTimer) ) {
      leds_toggle(LEDS_RED);
      etimer_set(&ledTimer, MY_CLOCK_SECOND);
    }

    if( status_channel_scan && etimer_expired(&scanTimer) ) {
      // If a packet has been received on the channel, stay there
      // But: if the CRC was broken, we need three packets to stay there
      if( stats_packets_on_channel_crc < 1 && stats_packets_on_channel_nocrc < 3 ) {
        // SLIP_PRINTF("CRC %u, NOCRC %u \n", stats_packets_on_channel_crc, stats_packets_on_channel_nocrc);

        // Our new channel is:
        stats_channel = (stats_channel + 1) % (MAX_CHANNEL + 1);

        if( stats_channel < MIN_CHANNEL ) {
          stats_channel = MIN_CHANNEL;
        }

        // Reset the counters
        stats_packets_on_channel_crc = 0;
        stats_packets_on_channel_nocrc = 0;

        // Set the channel in the radio
        cc2420_set_channel(stats_channel);
   
        // Inform the PC
        slipSendShortMessage(SLIP_TYPE_CHANNEL, stats_channel);
        
        // And restart the timer for the next switch
        etimer_set(&scanTimer, CHANNEL_SCAN_TIME);
      } else {
        // We have found a channel with traffic, staying here
        status_channel_scan = 0;

        // Inform the PC
        slipSendShortMessage(SLIP_TYPE_CHANNEL_SCAN_STOP, 0);
        slipSendShortMessage(SLIP_TYPE_CHANNEL, stats_channel);
      }
    }

    if( ev == serial_buffer_event ) {
      slipWorkIrqBuffer();
    }
  }

  PROCESS_END();
}
/*---------------------------------------------------------------------------*/
int slipIncomingMessage(unsigned char * buffer, u8_t length) {
  u8_t Type = buffer[0];

  if( Type == SLIP_TYPE_CHANNEL ) {
    if( buffer[length-1] == buffer[0] ) {
      // The message only contains the type, ignore it
      return -1;
    }

    if( buffer[length - 1] < MIN_CHANNEL ||
        buffer[length - 1] > MAX_CHANNEL ) {
      // Channel invalid, ignore
      return -1;
    }

    // The PC wants us to switch the channel
    stats_channel = buffer[length-1];
    
    // We set the channel in the radio
    cc2420_set_channel(stats_channel);

    if( status_channel_scan ) {
      // Abort the channel scan (if one is running)
      status_channel_scan = 0;
      
      // And inform the PC
      slipSendShortMessage(SLIP_TYPE_CHANNEL_SCAN_STOP, 0);
    }

    // And send a confirmation to the PC
    slipSendShortMessage(SLIP_TYPE_CHANNEL, stats_channel);
  } else if( Type == SLIP_TYPE_CHANNEL_SCAN_START ) {
    // PC asks us to restart the channel scan
    stats_packets_on_channel_crc = 0;
    stats_packets_on_channel_nocrc = 0;

    // Therefore: start on channel 0
    cc2420_set_channel(0);
    stats_channel = 0;

    // Start the timer
    status_channel_scan = 1;
    etimer_set(&scanTimer, CHANNEL_SCAN_TIME);

    // And send a confirmation message
    slipSendShortMessage(SLIP_TYPE_CHANNEL_SCAN_START, 0);
  } else if( Type == SLIP_TYPE_STOP ) {
    // Stop the actual delivery of packets to the PC
    status_sniffing = 0;
    
    // Send a confirmation that the status_sniffing is stopped
    slipSendShortMessage(SLIP_TYPE_STOP, 0); 
  } else if( Type == SLIP_TYPE_START ) {
    // Start the delivery of sniffed packets to the PC
    status_sniffing = 1;

    // Send a confirmation that the status_sniffing is started
    slipSendShortMessage(SLIP_TYPE_START, 0);
  } else if( Type == SLIP_TYPE_RSSI_START ) {

  } else if( Type == SLIP_TYPE_RSSI_STOP ) {

  } else {
    // The message seems to be broken - ignore it  
  }

  return 1;
}
