#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#include "crc16.h"
#include "ginseng-slip.h"
#include "serial_handler.h"
#include "udp_server.h"
#include "main.h"

enum {
  STATE_OK = 0,
  STATE_ESC = 1,
  STATE_NOK = 2,
};

unsigned char slip_in_buffer[GINSENG_SLIP_BUFFER];
int           slip_in_length;
int           slip_state;

static void (* serialSendChar)(unsigned char c) = NULL;

/*---------------------------------------------------------------------------*/
void slipSendEnd() {
  if( serialSendChar == NULL ) {
    return;
  }

  serialSendChar(SLIP_END);
}
/*---------------------------------------------------------------------------*/
void slipSendChar(unsigned char c) {
  c = c & 0xFF;

  if( serialSendChar == NULL ) {
    return;
  }
  
  if( c == SLIP_ESC ) {
    serialSendChar(SLIP_ESC);
    c = SLIP_ESC_ESC;
  } else if( c == SLIP_END ) {
    serialSendChar(SLIP_ESC);
    c = SLIP_ESC_END;    
  }

  serialSendChar(c);
}
/*---------------------------------------------------------------------------*/
void slipSendBuffer(unsigned char * buffer, int length) {
  int i;

  for(i=0; i<length; i++) {
    slipSendChar(buffer[i]);
  }
  
  slipSendEnd();
}
/*---------------------------------------------------------------------------*/
void slipSetOutput(void (*c)(unsigned char c)) {
  serialSendChar = c;
}
/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
int slipInputCharacter(unsigned char c) {
  c = 0xFF & c;

  switch(slip_state) {
    case STATE_OK:
      if( c == SLIP_ESC ) {
        // If we are reading data and we find an ESC, we need to discard the current char
        slip_state = STATE_ESC;
        return 0;
      } else if( c == SLIP_END ) {
        // END means a full packet has been received
        slipIncomingMessage(slip_in_buffer, slip_in_length);

        slip_in_length = 0;
        memset(slip_in_buffer, 0, GINSENG_SLIP_BUFFER);

        return 0;
      }
      break;

    case STATE_ESC:
      if( c == SLIP_ESC_END ) {
        // The sender escaped an SLIP_END
        c = SLIP_END;
        slip_state = STATE_OK;
      } else if( c == SLIP_ESC_ESC ) {
        // The sender escaped an SLIP_ESC
        c = SLIP_ESC;
        slip_state = STATE_OK;
      } else {
        // Looks like we are out if sync
        slip_state = STATE_NOK;
        return 0;
      }
      break;
  
    case STATE_NOK:
      if( c == SLIP_END ) {
        // If we do not know the state and find an END, we are synced
        slip_state = STATE_OK;

        slip_in_length = 0;
        memset(slip_in_buffer, 0, GINSENG_SLIP_BUFFER);
      }
      return 0;
  }
  
  slip_in_buffer[slip_in_length] = c; 
  slip_in_length++;

  if( slip_in_length > GINSENG_SLIP_BUFFER ) {
    printf("Buffer size exceeded\n");
    slip_state = STATE_NOK;
    slip_in_length = 0;
    memset(slip_in_buffer, 0, GINSENG_SLIP_BUFFER);
  }  

  return 1;
}
/*---------------------------------------------------------------------------*/
void slipInit() {
  serialSendChar = NULL;

  slip_state = STATE_NOK;
  slip_in_length = 0;
  memset(slip_in_buffer, 0, GINSENG_SLIP_BUFFER);

  slipSetOutput(serial_sendChar);

  slipSendEnd(); 
}
/*---------------------------------------------------------------------------*/
void slipSendShortMessage(u8_t type, u8_t payload) {
  slipSendChar(type);
  
  if( type == SLIP_TYPE_CHANNEL ) {
    slipSendChar(payload);
  }  

  slipSendEnd();
}
/*---------------------------------------------------------------------------*/
