#include "contiki.h"
#include "dev/uart1.h"
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include "ringbuf.h"
#include "net/rime.h"

#include "ginseng-slip.h"
#include "sniffer.h"

/*---------------------------------------------------------------------------*/
enum {
  STATE_OK = 0,
  STATE_ESC = 1,
  STATE_NOK = 2,
};

u8_t  slip_in_buffer[GINSENG_SLIP_BUFFER];
u8_t  slip_in_length;
u8_t  slip_state;

static struct ringbuf
      slip_irq_ring;
u8_t  slip_irq_buffer[64];
static void (* serialSendChar)(unsigned char c) = NULL;
/*---------------------------------------------------------------------------*/
void slip_hexdump(char * buffer, u8_t length) {
  u8_t h;
  slipSendChar(SLIP_TYPE_PRINTF);
  printf("Hex: ");

  for(h=0; h<length; h++) {
    printf("%02X ", buffer[h]);
  }
  printf("\n");
  
  slipSendEnd();
}
/*---------------------------------------------------------------------------*/
void slipSendEnd() {
  serialSendChar(SLIP_END);
}
/*---------------------------------------------------------------------------*/
void slipSendChar(unsigned char c) {
  c = c & 0xFF;

  if( serialSendChar == NULL ) {
    return;
  }
  
  if( c == SLIP_ESC ) {
    serialSendChar(SLIP_ESC);
    c = SLIP_ESC_ESC;
  } else if( c == SLIP_END ) {
    serialSendChar(SLIP_ESC);
    c = SLIP_ESC_END;    
  }
  
  serialSendChar(c);
}
/*---------------------------------------------------------------------------*/
void slipSendBuffer(unsigned char * buffer, u8_t length) {
  u8_t i;

  for(i=0; i<length; i++) {
    slipSendChar(buffer[i]);
  }
  
  slipSendEnd();
}
/*---------------------------------------------------------------------------*/
void slipSendShortMessage(u8_t type, u8_t payload) {
  slipSendChar(type);
  
  if( type == SLIP_TYPE_CHANNEL ) {
    slipSendChar(payload);
  }  

  slipSendEnd();
}
/*---------------------------------------------------------------------------*/
void slipSetOutput(void (*c)(unsigned char c)) {
  serialSendChar = c;
}
/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
int slipUartInterrupt(unsigned char c) {
  process_post(&sniffer_process, serial_buffer_event, NULL);
  return ringbuf_put(&slip_irq_ring, c);
}
/*---------------------------------------------------------------------------*/
int slipWorkIrqBuffer() {
  while( ringbuf_elements(&slip_irq_ring) > 0 ) {
    slipInputCharacter(ringbuf_get(&slip_irq_ring));
  }

  return 1;
}
/*---------------------------------------------------------------------------*/
int slipInputCharacter(unsigned char c) {
  c = 0xFF & c;

  switch(slip_state) {
    case STATE_OK:
      if( c == SLIP_ESC ) {
        // If we are reading data and we find an ESC, we need to discard the current char
        slip_state = STATE_ESC;
        return 0;
      } else if( c == SLIP_END ) {
        // END means a full packet has been received
        slipIncomingMessage(slip_in_buffer, slip_in_length);

        slip_in_length = 0;
        memset(slip_in_buffer, 0, GINSENG_SLIP_BUFFER);

        return 0;
      }
      break;

    case STATE_ESC:
      if( c == SLIP_ESC_END ) {
        // The sender escaped an SLIP_END
        c = SLIP_END;
        slip_state = STATE_OK;
      } else if( c == SLIP_ESC_ESC ) {
        // The sender escaped an SLIP_ESC
        c = SLIP_ESC;
        slip_state = STATE_OK;
      } else {
        // Looks like we are out if sync
        slip_state = STATE_NOK;
        return 0;
      }
      break;
  
    case STATE_NOK:
      if( c == SLIP_END ) {
        // If we do not know the state and find an END, we are synced
        slip_state = STATE_OK;

        slip_in_length = 0;
        memset(slip_in_buffer, 0, GINSENG_SLIP_BUFFER);
      }
      return 0;
  }
  
  slip_in_buffer[slip_in_length] = c; 
  slip_in_length++;

  if( slip_in_length > GINSENG_SLIP_BUFFER ) {
    SLIP_PRINTF("Buffer size exceeded\n");
    slip_state = STATE_NOK;
    slip_in_length = 0;
    memset(slip_in_buffer, 0, GINSENG_SLIP_BUFFER);
  }  

  return 1;
}
/*---------------------------------------------------------------------------*/
void slipInit() {
  serialSendChar = NULL;

  slip_state = STATE_NOK;
  slip_in_length = 0;
  memset(slip_in_buffer, 0, GINSENG_SLIP_BUFFER);

  ringbuf_init(&slip_irq_ring, slip_irq_buffer, 64);

  // Reinit the serial port at a higher speed!
  uart1_init(BAUD2UBR(921600));

  uart1_set_input(slipUartInterrupt);
  slipSetOutput(uart1_writeb);

  slipSendEnd(); 
 
  SLIP_PRINTF("Slip ready\n");
  slipSendEnd(); 
}
/*---------------------------------------------------------------------------*/
int putchar(int c) {
  slipSendChar(c);
  return c;
}
