#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <termios.h>

#include "main.h"
#include "crc16.h"
#include "serial_handler.h"
#include "ginseng-slip.h"
#include "udp_server.h"
#include "ethernet_handler.h"

int last_msg;

int status_scan;
int status_sniffing;
int status_rssi;

int current_channel;
int regenerate_crc;

void IntHandler(int signum) {
  RUNNING = 0;
}

int main (int argc, char ** argv) {
  int h, len, ret;
  char * serial_device = "/dev/ttyUSB0";
  char * udp_host = "127.0.0.1";
  int    udp_port = 0;
  int    channel = 0;
  char * ethernet_device = NULL;
  int    ethernet_type   = 0x809a;
  
  RUNNING         = 1;
  regenerate_crc  = 1;

  signal(SIGINT, IntHandler);

  for(h=1; h<argc; h++) {
    len = strlen(argv[h]);

   if( len == 8 && strncmp(argv[h], "--serial", 8) == 0 && (h+1) < argc) {
      serial_device = argv[h+1];
      h++;
   } else if( len == 9 && strncmp(argv[h], "--udphost", 9) == 0 && (h+1) < argc) {
      udp_host = argv[h+1];
      h++;
   } else if( len == 9 && strncmp(argv[h], "--udpport", 9) == 0 && (h+1) < argc) {
      udp_port = atoi(argv[h+1]);
      h++;
   } else if( len == 9 && strncmp(argv[h], "--channel", 9) == 0 && (h+1) < argc) {
      channel = atoi(argv[h+1]);
      h++;
   } else if( len == 9 && strncmp(argv[h], "--ethtype", 9) == 0 && (h+1) < argc) {
      ethernet_type = atoi(argv[h+1]);
      h++;
   } else if( len == 8 && strncmp(argv[h], "--ginmac", 8) == 0 ) {
      ethernet_type = 0x809b;
   } else if( len == 11 && strncmp(argv[h], "--ethdevice", 11) == 0 && (h+1) < argc) {
      ethernet_device = argv[h+1];
      h++;
   } else if( len == 7 && strncmp(argv[h], "--nocrc", 7) == 0) {
      regenerate_crc = 0;
   } else {
      printf("%s - GINSENG IEEE 802.15.4 Packet Sniffer\n", argv[0]);
      printf("(c) 2010 Wolf-Bastian Poettner <poettner@ibr.cs.tu-bs.de.de>\n");
      printf("\n");
      printf("  --serial    Specify the serial device to use [%s]\n", serial_device);
      printf("\n");
      printf("  --channel   Specify the 802.15.4 Channel to use [scan]\n");
      printf("  --nocrc     Do not regenerate the FCS (for newer Wiresharks)\n");
      printf("\n");
      printf("  --ethdevice Specify the device used for raw ethernet packets [%s]\n", ethernet_device); 
      printf("  --ethtype   Specify the packet type used in the ethernet frames [%04X]\n", ethernet_type);
      printf("  --ginmac    Switch the ethernet type to ginmac [%04X]\n", 0x809b);
      printf("\n");
      printf("  --udphost   Specify the UDP host to send data to [%s]\n", udp_host);
      printf("  --udpport   Specify the UDP port to send data to [%u]\n", udp_port);
      printf("\n");
      printf("  --help      Display this screen\n");
      exit(1);
    }
  }

  ret = serial_init(serial_device);
  if( ret < 0 ) {
    printf("ERROR: serial_init failed\n");
    exit(1);
  }

  if( udp_port > 0 ) {
    udp_init(udp_host, udp_port);
  }

  if( ethernet_device != NULL ) {
    ethernet_init(ethernet_device, ethernet_type);
  }

  snifferInit();

  if( channel != 0 ) {
    // Change the channel of the sniffer, abort the scan
    snifferChangeChannel(channel, 1);
  }

  keyboard_set_nonblock(); 

  while(RUNNING) {
    snifferService();
    keyboard_service();
    usleep(1);
  }
 
  serial_deinit();
  udp_deinit();
  ethernet_deinit();
 
  keyboard_set_block();

  return 1;
}
/*---------------------------------------------------------------------------*/
int keyboard_key_pressed() {
  struct timeval my_timeval;
  fd_set key_fd;

  my_timeval.tv_sec = 0;
  my_timeval.tv_usec = 0;

  FD_ZERO(&key_fd);
  FD_SET(STDIN_FILENO, &key_fd);

  select(STDIN_FILENO + 1, &key_fd, NULL, NULL, &my_timeval);

  if( FD_ISSET(STDIN_FILENO, &key_fd) ) {
    return 1;
  } else {
    return 0;
  }
}
/*---------------------------------------------------------------------------*/
int keyboard_set_nonblock() {
  struct termios terminal_state;

  // Get the current attributes of the terminal
  tcgetattr(STDIN_FILENO, &terminal_state);

  // Disable canoncial mode
  terminal_state.c_lflag &= ~ICANON;
    
  // Set the buffer to 1 character
  terminal_state.c_cc[VMIN] = 1;
  
  // Set the terminal attributes
  tcsetattr(STDIN_FILENO, TCSANOW, &terminal_state);

  return 1;
}
/*---------------------------------------------------------------------------*/
int keyboard_set_block() {
  struct termios terminal_state;

  // Get the current attributes of the terminal
  tcgetattr(STDIN_FILENO, &terminal_state);

  // Enable canoncial mode
  terminal_state.c_lflag |= ICANON;
    
  // Set the terminal attributes
  tcsetattr(STDIN_FILENO, TCSANOW, &terminal_state);

  return 1;
}
/*---------------------------------------------------------------------------*/
int snifferInit() {
  status_sniffing = 1;
  status_scan = 0;
  status_rssi = 0;

  return 1;
}
/*---------------------------------------------------------------------------*/
int snifferService() {
  serial_service();

  return 1;
}
/*---------------------------------------------------------------------------*/
int slipIncomingMessage(unsigned char * message, int length) {
  int i;
  float ts_ms;
  
  int type = message[0];

  if( type == SLIP_TYPE_SNIFFED ) {
    // Sniffed raw packet
    int ts     = ((message[1] & 0xFF) << 8) + message[2];
    int diff;
    int channel = message[3] & 0xFF;
    short len  = message[4] & 0xFF;
    short rssi = (message[5 + len - 2] + 55) & 0xFF;

    status_sniffing = 1;
    current_channel = channel; 
 
    // Calculate the offset to the last timestamp
    if( ts < last_msg ) {
      // It rolled over on the node
      diff = (65535 - last_msg) + ts;
    } else {
      diff = ts - last_msg;
    }

    last_msg = ts;

    ts_ms = (float) diff * (float) 1000000 / (float) 32768;

    printf("+%8.0f us [L%2u, R%2u, C%2u]: ", ts_ms, len, rssi, channel);
    for(i=5; i<length; i++) {
      printf("%02X ", message[i]);
      if( i == (1 + 5) || i == (2 + 5) || i == (4 + 5) || i == (6 + 5) || i == (8 + 5) || i == (length - 3) ) {
        printf("| ");
      }
    }
    
    if( message[5 + len - 1] & CC2420_CRC_OK ) {
      // To send it via UDP, we have to repair the checksum so make wireshark happy
      if( regenerate_crc ) {
        int crc = crc16_data(&message[5], len - 2, 0);
        message[5 + len - 2] = (crc & 0x00FF) >> 0;
        message[5 + len - 1] = (crc & 0xFF00) >> 8;
      } 
      printf("[CRC OK]");
    } else {  
      printf("[CRC FAILED]");
    }
    printf("\n");
   
    udp_send((char *) &message[5], len);
    ethernet_send((char *) &message[5], len);
  } else if( type == SLIP_TYPE_PRINTF ) {
    printf("[NODE] %s", &message[1]);
  } else if( type == SLIP_TYPE_RSSI ) {
    
  } else if( type == SLIP_TYPE_CHANNEL ) {
    printf("[NODE] Channel %u \n", message[1]);
    current_channel = message[1];

  } else if( type == SLIP_TYPE_CHANNEL_SCAN_START ) {
    status_scan = 1;
    printf("[NODE] Starting channel scan...\n");
  } else if( type == SLIP_TYPE_CHANNEL_SCAN_STOP ) {
    status_scan = 0;
    printf("[NODE| Channel scan stopped\n");
  } else if( type == SLIP_TYPE_RSSI_START ) {
    status_rssi = 1;
  } else if( type == SLIP_TYPE_RSSI_STOP ) {
    status_rssi = 0; 
  } else if( type == SLIP_TYPE_START ) {
    status_sniffing = 1;
    printf("[NODE] Start sniffing mode\n");
  } else if( type == SLIP_TYPE_STOP ) {
    status_sniffing = 0;
    printf("[NODE] Stop sniffing mode\n");
  } 

  return 1;
}
/*---------------------------------------------------------------------------*/
int snifferStartSniffing() {
  int tries = 0;

  // The serial connection to the node is unreliable, as long as packets are 
  // being sniffed. Therefore, we may have to send requests multiple times
  // until they are received correctly 
  while(RUNNING && !status_sniffing && tries < 100) {
    slipSendShortMessage(SLIP_TYPE_START, 0);
    usleep(10000);
    serial_service();
    tries ++;
  }

  if( status_sniffing ) {
    return 1; 
  }

  return -1;
}
/*---------------------------------------------------------------------------*/
int snifferStopSniffing() {
  int tries = 0;

  // The serial connection to the node is unreliable, as long as packets are 
  // being sniffed. Therefore, we may have to send requests multiple times
  // until they are received correctly 
  while(RUNNING && status_sniffing && tries < 100) {
    slipSendShortMessage(SLIP_TYPE_STOP, 0);
    usleep(10000);
    serial_service();
    tries ++;
  }

  if( !status_sniffing ) {
    return 1; 
  }

  return -1;
}
/*---------------------------------------------------------------------------*/
int snifferSetChannel(int channel) {

  int tries = 0;

  // The serial connection to the node is unreliable, as long as packets are 
  // being sniffed. Therefore, we may have to send requests multiple times
  // until they are received correctly 
  while(RUNNING && current_channel != channel && tries < 100) {
    slipSendShortMessage(SLIP_TYPE_CHANNEL, channel);
    usleep(10000);
    serial_service();
    tries ++;
  }

  if( current_channel == channel ) {
    return 1; 
  }

  return -1;
}  
/*---------------------------------------------------------------------------*/
int snifferChangeChannel(int channel, int sniffing) {
  int ret;

  if( current_channel == channel ) {
    return 0;
  }

  ret = snifferStopSniffing();

  if( ret != 1 ) {
    return ret;
  } 

  ret = snifferSetChannel(channel);

  if( ret != 1 ) {
    return ret;
  }

  if( sniffing ) {
    ret = snifferStartSniffing();

    if( ret != 1 ) {
      return ret;
    }
  }

  return 1;
}
/*---------------------------------------------------------------------------*/
int snifferStartScan() {
  int tries = 0;

  // The serial connection to the node is unreliable, as long as packets are 
  // being sniffed. Therefore, we may have to send requests multiple times
  // until they are received correctly 
  while(RUNNING && !status_scan && tries < 100) {
    slipSendShortMessage(SLIP_TYPE_CHANNEL_SCAN_START, 0);
    usleep(10000);
    serial_service();
    tries ++;
  }

  if( status_scan ) {
    return 1; 
  }

  return -1;

}
/*---------------------------------------------------------------------------*/
int keyboard_service() {
  char c;
  char input[5];
  int channel;
  char * ret;

  if( keyboard_key_pressed() ) {
    c = fgetc(stdin);

    if( c == 'q' ) {
      // Quit
      RUNNING = 0;
    } else if( c == 's' ) {
      // Start Scan
      snifferStartScan();
    } else if( c == 'c' ) {
      // Change channel
      fputs("Channel? ", stdout);
      fflush(stdout);
      ret = fgets(input, sizeof(input), stdin);

      if( ret != NULL ) {
        channel = atoi(input);

        if( channel < MIN_CHANNEL || channel > MAX_CHANNEL ) {
          printf("[ERROR] Channel %u not supported\n", channel);
          return 0;
        }

        snifferChangeChannel(channel, status_sniffing);
      }
    }
  }

  return 1;
}       
/*---------------------------------------------------------------------------*/
