/*
 * Created on 13.08.2004
 *
 */
package mapper.GUIComponents;

import java.util.Iterator;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.swing.JComponent;
import javax.swing.border.TitledBorder;

import java.awt.GraphicsConfiguration;
import java.awt.GraphicsEnvironment;
import java.awt.GraphicsDevice;
import java.awt.Graphics2D;
import java.awt.Graphics;
import java.awt.RenderingHints;
import java.awt.Image;
import java.awt.Dimension;
import java.awt.Color;
import java.awt.image.BufferedImage;

import java.io.File;
import java.io.IOException;

import java.awt.geom.AffineTransform;

import javax.imageio.ImageIO;

import mapper.DataStrukture.Node;
import mapper.DataStrukture.Position;
import mapper.DataStrukture.MovementField;
import mapper.DataStrukture.ValueNew;
import mapper.DataStrukture.ValueTable;
import mapper.DataStrukture.PositionArray;
import mapper.DataStrukture.PathLine;

/**
 * Klasse auf dem die graphischen Darstellung der Szenariokarte erstellt wird.
 * 
 * @author Emanuel Eden
 *  
 */
public class PaintingArea extends JComponent {

    // Image in der Originalgrösse
    protected BufferedImage _image;
    // Zoom Image
    protected BufferedImage _transImage;

    private ValueTable _valueTable = new ValueTable();
    private MovementField _movementField = null;
    private Node _node = null;
    private PathLine _pline;

    // Wird die Painting Area gezeichnet
    private boolean _paintingAreaIsDefind = false;
    // Gibt es eine Karte in der PaintingArea
    private boolean _mapInPaintingArea = false;
    // Soll Karte angezeigt werden
    private boolean _hidePictureInPaintingArea = false;
    // Gibt es ein markietes Objekt
    private boolean _marked = false;
    // Sind Polygone in der HashMap
    private boolean _paintRectangle = false;
    // Zeichne Pfad ein
    private boolean _paintPath = false;
    // Momentaner Zoom-Faktor
    private double _currentZoom = 1.0;
    // Zoom
    private double _zoom = 1.0;

    Color blue = new Color(118, 129, 196);
    Color dblue = new Color(154, 167, 255);
    Color red = new Color(255, 0, 0);
    Color black = new Color(0, 0, 0);
    Color white = new Color(255, 255, 255);

    /**
     * Initialisierung der PaintingArea
     */
    public PaintingArea() {

        super();
        _mapInPaintingArea = false;
        _paintingAreaIsDefind = false;
        setBorder(new TitledBorder("No Map specified"));
    }

    /**
     * Liefert immer die Grafikkonfiguration der BasisConfiguration zurück.
     */
    public static GraphicsConfiguration getDefaultConfiguration() {

        GraphicsEnvironment ge = GraphicsEnvironment
                .getLocalGraphicsEnvironment();
        GraphicsDevice gd = ge.getDefaultScreenDevice();
        return gd.getDefaultConfiguration();
    }

    /**
     * Legt die Dimension der PaintingArea fest.
     * 
     * @param valueNew
     *            Informationen über das Szenario
     */
    public void setDimension(ValueNew valueNew) {

        this.setBorder(null);
        _valueTable.setValueNew(new ValueNew(valueNew));
        _paintingAreaIsDefind = true;
    }

    /**
     * Legt die Karte für die PaintingArea fest und berechnet die Differenz
     * zwischen PaintingArea und des momentanen Map-Pictures.
     * 
     * @param file Imagefile auf dem die Karte gezeichnet ist.
     */
    public void setImageFile(File file) {

        ValueNew valueNew = new ValueNew(_valueTable.getValueNew());
        Position relationship = new Position();
        setImageToPaintingArea(file);
        _mapInPaintingArea = true;
        paintComponent(this.getGraphics());

        relationship.setX(_image.getWidth()
                / _valueTable.getValueNew().getDimension().getX());
        relationship.setY(_image.getHeight()
                / _valueTable.getValueNew().getDimension().getY());
        _valueTable.getValueNew().setRelationship(relationship);
    }

    /**
     * Berechnet den Zoom neu und übergigt sie ans _transImage.
     * 
     * @param zoom Zoomfaktor
     */
    public void zoom(double zoom) {

        _currentZoom = zoom;
        //System.out.println("-----> Zoom: "+ zoom);
        if (_mapInPaintingArea) {
            flush(_transImage);
            _transImage = null;
            _transImage = resize(_image, zoom);
            invalidate();
        }
    }

    /**
     * Versteckt das Kartenimage
     *
     */
    public void hideImage() {

        _hidePictureInPaintingArea = !_hidePictureInPaintingArea;
    }

    /**
     * Übergibt ein unfertiges MovementField, damit bei seiner Generierung die 
     * Polygonknoten in der PaintingArea angezeigt werden können oder, damit es
     * als markiert angezeigt werden kann.
     * 
     * @param movementField Übergibt ein neu zu generierendes MovementField. Das
     * MovementField wird erst in die PaintingArea gezeichnet, wenn 
     * <code>setMarked</code> auf true gesetzt wird.
     */
    public void setMovementField(MovementField movementField) {

        _movementField = movementField;
    }
    
    /**
     * Setzt ein neues Node, als markiertes Node ein. Das Node wird erst in die
     * PaintingArea gezeichnet, wenn <code>setMarked</code> auf true gesetzt wird.
     * 
     * @param node zu markierendesNode
     */
    public void setNode(Node node) {

        _node = node;
    }
    
    /**
     * Setzt die Markierung eines Objektes in der PaintingArea ein. Falls ein 
     * <code>true<code> eingesetzt wird, befindet sich ein Objekt in der 
     * PaintingArea. Ein neues markiertes Objekt kann mit <code>setMovementField
     * </code> oder mit <code>setNode</code> eingesetzt werden.
     * 
     * @param marked Kennzeichnet ein belibiges Objekt als markiert
     */
    public void setMarked(boolean marked) {

        _marked = marked;
    }

    /**
     * Liefert die momentane Dimension zurück, damit z.B die Ausmaße für die 
     * JScroll richtig eingefügt werden können.
     * 
     * @return Dimension Ausmaße der PaintingArea
     */
    public Dimension getPreferredSize() {

        if (_paintingAreaIsDefind)
            if (_mapInPaintingArea)
                return new Dimension(_transImage.getWidth(), _transImage
                        .getHeight());
            else {
                return new Dimension(
                        (int) (_valueTable.getValueNew().getDimension().getX() * computeZRX()),
                        (int) (_valueTable.getValueNew().getDimension().getY() * computeZRY()));
            }
        return new Dimension(100, 100);
    }

    /**
     * Setzt ein neues Value Table in die PaintingArea
     * 
     * @param valueTable neues ValueTable
     */
    public void setValueTable(ValueTable valueTable) {

        _valueTable = valueTable;
    }

    /**
     * Liefert die Differenz zwischen Karte und Szenario Dimension
     * 
     * @return Position der RelationShip Differenz
     */
    public Position getRelationship() {

        return _valueTable.getValueNew().getRelationship();
    }

    /**
     * Setzt eine Linie in die PaintingArea. Dies wird verwendet um den letzten
     * Abschnitt eines Paths zwischen markiertem Objekt (Stoppunkt des letzten 
     * <code>TimeSlots</code> im <code>TimeScheduler</code>) des markierten 
     * Objektes und dem Cursor zu setzten. 
     * 
     * @param pline Line zwischen TimeSlot Endpunkt und Cursor
     */
    public void setPath(PathLine pline) {

        _paintPath = true;
        _pline = pline;
    }

    /**
     * Beendet die Eingabe für dem Path eines <code>TimeSchedulers TimeSlots</code>
     * und setzt alle Werte auf default Werte.
     */
    public void finishPath() {

        _paintPath = false;
        _pline = new PathLine(null, null);
    }

    /**
     * Zeichnet ein Rectangle in der Phase in der es noch nicht seine endgültige
     * Position bekommen hat. (Themporäres Rectangle)
     * 
     * @param position1 Anfangsposition
     * @param position2 momentane Cursorposition
     */
    public void setRectangle(Position position1, Position position2) {

        _paintRectangle = true;
        _pline = new PathLine(position1, position2);
    }

    /**
     * Beendet das Zeichnen eines Rectangles und setzt alle Werte auf default.
     */
    public void finishRectangle() {

        _paintRectangle = false;
        _pline = new PathLine(null, null);
    }

    /**
     * Läd das Image aus der Datei in den Objektpuffer. Bei dem Image handelt es 
     * sich um die definierte Hintergrundkarte des Szenarios.
     * 
     * @param image Location und Name des zu landenden Images
     * @return BufferedImage Liefert das geladene Image zurück.
     */
    private BufferedImage loadImage(File image) {

        try {
            BufferedImage im = ImageIO.read(image);
            System.out.println("-----> Image : Width: " + im.getWidth()
                    + ", Height: " + im.getHeight());
            BufferedImage copy = resize(im, 1);
            flush(im);
            return copy;
        } catch (IOException e) {
            System.out.println("Load Image error for " + image + ":\n" + e);
            return null;
        }
    }

    /**
     * Ändert die Dimensionen des Images, sofern der Zoomfaktor verändert wurde.
     * 
     * @param src Image dessen Zoomfaktor geändert werden soll
     * @param zoom Zoomfaktor der auf das Image angewendet werden soll
     * @return BufferedImage Liefert das neu berechnete Image zurück.
     */
    private BufferedImage resize(BufferedImage src, double zoom) {

        if (src == null)
            return null;
        int transparency = src.getColorModel().getTransparency();
        int width = (int) (src.getWidth() * zoom);
        int height = (int) (src.getHeight() * zoom);
        GraphicsConfiguration gc = getDefaultConfiguration();
        BufferedImage dest = gc.createCompatibleImage(width, height,
                transparency);
        Graphics2D g2d = dest.createGraphics();
        g2d.setRenderingHint(RenderingHints.KEY_INTERPOLATION,
                RenderingHints.VALUE_INTERPOLATION_NEAREST_NEIGHBOR);
        AffineTransform xform = (zoom == 1d) ? null : AffineTransform
                .getScaleInstance(zoom, zoom);
        g2d.drawRenderedImage(src, xform);
        g2d.dispose();
        return dest;
    }

    /**
     * Gibt das Image auf der PaintingArea aus oder genauer erlaubt der PaintingArea
     * es zu zeichnen
     * 
     * @param file Location der Karte
     */
    private void setImageToPaintingArea(File file) {

        try {
            _image = loadImage(file);
            _transImage = _image;
        } catch (Exception e) {
            System.out.println(e.getMessage());
        }
        _mapInPaintingArea = true;
    }

    /**
     * Löscht das Image (Bereinigt das Image-Objekt). Wird vor der Neuberechnung 
     * ausgeführt.
     * @param image Das Image das bereinigt werden soll
     */
    private static void flush(Image image) {

        try {
            if (image != null)
                image.flush();
        } catch (NullPointerException e) {
            System.err.println("Error with the image:flush");
            e.printStackTrace();
        }
    }
    
    /**
     * PaintingArea Ausgabe. Zeichnen der Komponenten.
     */
    protected void paintComponent(Graphics g) {

        if (_paintingAreaIsDefind)
            drawPaintingArea(g);

        drawMovementField(g);
        if (_movementField != null && _marked == true)
            drawMarkedPolygon(g);
        drawNodes(g);
        if (_node != null && _marked == true)
            drawMarkedNode(g);
        if (_movementField != null && _marked == false)
            drawMovementFieldCreation(g);
        if (_paintRectangle == true)
            drawRectangle(g);
        if (_paintPath == true)
            drawPath(g);
    }

    /**
     * Zeinet die Ausmaße der PaintingArea in der definierten Grösse des Szenarios
     * in der Farbe weiß.
     * 
     * @param g Grafik-Kontext
     */
    private void drawPaintingArea(Graphics g) {

        if (_mapInPaintingArea && !_hidePictureInPaintingArea) {
            ((Graphics2D) g).drawRenderedImage(_transImage, null);
        } else {
            g.setColor(white);
            g.fillRect(0, 0, (int) (_valueTable.getValueNew().getDimension()
                    .getX() * computeZRX()), (int) (_valueTable.getValueNew()
                    .getDimension().getY() * computeZRY()));
        }
    }
    
    /**
     * Zeichnet den Path eines Nodes oder MovementFields.
     * 
     * @param g Grafik-Kontexte
     */
    private void drawPath(Graphics g) {

        g.setColor(black);
        g.drawLine((int) _pline.getSource().getX(), (int) _pline.getSource()
                .getY(), (int) _pline.getSink().getX(), (int) _pline.getSink()
                .getY());
    }

    /**
     * Zeichnet das unfertige Rectangle eines MovementFields.
     * 
     * @param g Grafik-Kontexte
     */
    private void drawRectangle(Graphics g) {

        g.setColor(black);
        g.drawLine((int) _pline.getSource().getX(), (int) _pline.getSource()
                .getY(), (int) _pline.getSource().getX(), (int) _pline
                .getSink().getY());
        g.drawLine((int) _pline.getSource().getX(), (int) _pline.getSource()
                .getY(), (int) _pline.getSink().getX(), (int) _pline
                .getSource().getY());

        g.drawLine((int) _pline.getSource().getX(), (int) _pline.getSink()
                .getY(), (int) _pline.getSink().getX(), (int) _pline.getSink()
                .getY());
        g.drawLine((int) _pline.getSink().getX(), (int) _pline.getSource()
                .getY(), (int) _pline.getSink().getX(), (int) _pline.getSink()
                .getY());
    }

    /**
     * Zeichnet alle Nodes in der PaintingArea.
     * 
     * @param g Grafik-Kontexte
     */
    private void drawNodes(Graphics g) {

        Iterator iter = _valueTable.getNodes().iterator();
        while (iter.hasNext()) {
            Node node = (Node) iter.next();
            if (node.isRandomStartPoint() == true)
                continue;
            Position position = (Position) node.getPosition();
            g.setColor(black);
            g.drawOval((int) (position.getX() * computeZRX() - 3),
                    (int) (position.getY() * computeZRY()) - 3, 6, 6);
            g.setColor(blue);
            g.drawOval((int) (position.getX() * computeZRX() - 4),
                    (int) (position.getY() * computeZRY()) - 4, 8, 8);
            g.setColor(black);
            g.drawOval((int) (position.getX() * computeZRX() - 5),
                    (int) (position.getY() * computeZRY()) - 5, 10, 10);
            int key = node.getTimeScheduler().getFirstKey();
            while (key != -1) {
                Position p1 = node.getTimeScheduler().getTimeSlot(key)
                        .getStartPosition();
                Position p2 = node.getTimeScheduler().getTimeSlot(key)
                        .getStopPosition();
                g.drawLine((int) (p1.getX() * computeZRX()),
                        (int) (p1.getY() * computeZRY()),
                        (int) (p2.getX() * computeZRX()),
                        (int) (p2.getY() * computeZRY()));
                key = node.getTimeScheduler().getSuccessor(key);
            }
        }
    }

    /**
     * Zeichnet alle MovementFields in die PaintingArea 
     * 
     * @param g Grafik-Kontexte
     */
    private void drawMovementField(Graphics g) {

        HashMap movementFieldList = new HashMap(_valueTable
                .getMovementFieldList());
        Set set = movementFieldList.entrySet();

        Iterator iter = set.iterator();
        while (iter.hasNext()) {

            PositionArray points = new PositionArray(computeZRX(), computeZRY());
            Map.Entry element = (Map.Entry) iter.next();
            MovementField movementField = (MovementField) element.getValue();
            if (movementField.getStaticStatus())
                g.setColor(blue);
            else
                g.setColor(dblue);
            Iterator iterator = movementField.getNodes().iterator();
            while (iterator.hasNext()) {

                points.setPosition((Position) iterator.next());
            }
            g.fillPolygon(points.getPointsX(), points.getPointsY(), points
                    .getLength());
            int key = movementField.getTimeScheduler().getFirstKey();
            g.setColor(black);

            while (key != -1) {
                Position p1 = movementField.getTimeScheduler().getTimeSlot(key)
                        .getStartPosition();
                Position p2 = movementField.getTimeScheduler().getTimeSlot(key)
                        .getStopPosition();
                g.drawLine((int) (p1.getX() * computeZRX()),
                        (int) (p1.getY() * computeZRY()),
                        (int) (p2.getX() * computeZRX()),
                        (int) (p2.getY() * computeZRY()));
                key = movementField.getTimeScheduler().getSuccessor(key);
            }

        }

    }

    /**
     * Zeichnet ein unfertiges MovementField Objekt. Alle bereits definierten Knoten
     * und die Zuordnung zueinander werden gezeichnet.
     * 
     * @param g Grafik-Kontexte
     */
    private void drawMovementFieldCreation(Graphics g) {

        Position tmp = new Position();
        Iterator iter = _movementField.getNodes().iterator();
        while (iter.hasNext()) {
            g.setColor(red);
            Position position = (Position) iter.next();
            g.drawOval((int) (position.getX() * computeZRX()) - 3,
                    (int) (position.getY() * computeZRY()) - 3, 6, 6);
            if ((tmp.getX() != -1) && (tmp.getY() != -1)) {
                g.setColor(black);
                g.drawLine((int) (tmp.getX() * computeZRX()),
                        (int) (tmp.getY() * computeZRY()), (int) (position
                                .getX() * computeZRX()),
                        (int) (position.getY() * computeZRY()));
            }
            tmp = position;
        }
    }

    /**
     * Zeichnet ein Markiertes MovementField Objekt in der Farbe rot.
     * 
     * @param g Grafik-Kontexte
     */
    private void drawMarkedPolygon(Graphics g) {

        g.setColor(red);
        PositionArray points = new PositionArray(computeZRX(), computeZRY());
        Iterator iterator = _movementField.getNodes().iterator();
        while (iterator.hasNext()) {
            points.setPosition((Position) iterator.next());
        }
        g.fillPolygon(points.getPointsX(), points.getPointsY(), points
                .getLength());
        g.setColor(black);
        for (int i = 0; i < points.getLength(); i++)
            g.drawOval(points.getX(i) - 3, points.getY(i) - 3, 6, 6);
    }

    /**
     * Zeichnet ein markiertes Node in der Farbe rot. 
     * 
     * @param g Grafik-Kontexte
     */
    private void drawMarkedNode(Graphics g) {

        g.setColor(red);
        Position position = (Position) _node.getPosition();
        g.drawOval((int) (position.getX() * computeZRX() - 3), (int) (position
                .getY() * computeZRY()) - 3, 6, 6);
        g.drawOval((int) (position.getX() * computeZRX() - 4), (int) (position
                .getY() * computeZRY()) - 4, 8, 8);
        g.drawOval((int) (position.getX() * computeZRX() - 5), (int) (position
                .getY() * computeZRY()) - 5, 10, 10);
    }

    /**
     * Berechnet die tatsächlichen Ausmaße eines Objektes mittels des Zoom Faktors
     * und der Differenz zwischen Karte und Szenario Dimensionen.
     * 
     * @return X-Koordinate des Zoomfaktors
     */
    private double computeZRX() {

        return _currentZoom
                * _valueTable.getValueNew().getRelationship().getX();
    }
    
    /**
     * Berechnet die tatsächlichen Ausmaße eines Objektes mittels des Zoom Faktors
     * und der Differenz zwischen Karte und Szenario Dimensionen.
     * 
     * @return Y-Koordinate des Zoomfaktors
     */
    private double computeZRY() {

        return _currentZoom
                * _valueTable.getValueNew().getRelationship().getY();
    }
}