/*
 * Created on 14.08.2004
 */
package mapper.DataStrukture;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.ListIterator;

/**
 * Speichert alle Relevanten Daten über das Szenario ab, darunter die MovementFields,
 * die Nodes und merkt sich ob markierungen von Objekten vorliegen. 
 * 
 * @author Emanuel Eden
 *
 */
public class ValueTable {
	
	// Enthält die Basisinformationen eines Szenarios
	private ValueNew _valueNew = new ValueNew();
	// Beinhaltet die Nodes und MovementFields des Szenarios
	private HashMap _movementFieldList = new HashMap();
	private HashSet _nodeList = new HashSet();
	
	// Markierte Objekte, z.B fürs Kopieren oder Verschieben von Knoten und Polygonen
	private MovementField _markedMovementField;
	private Node _markedNode;
	private Position _markedPolygonNode;
	private Object _markedKey;
	private int _marked = -1;
	// Variablen zur Durchnummerieung der NameNumbers für den CpiceTree
	private int _numberOfMovementFields = 0;
	private int _numberOfNodes = 0;
	
	
	/**
	 * Fügt ein MovementField Element in die <code>_movementField</code> Liste
	 * und erhält eine inkremetelle Laufnummer, über den das Element identifiziert 
	 * werden kann.  Die Laufnummer ist der Key der HashMap und das <code>
	 * movementField</code> ist der Value-Wert. Das soeben eingefügte MovementField
	 * wird mit der soeben zugeordneten Laufnummer zurückgegeben, damit es an
	 * den ChoiceTree weiter gegeben werden kann.
	 * 
	 * @param movementField MovmentField, das in die <code>_movementField</code> 
	 * Liste eingefügt wird.
	 * @return Key, über den das Movementfield gehasht wird.
	 */
	public MovementField setElement(MovementField movementField) {
		
		movementField.setName(_numberOfMovementFields);
		_movementFieldList.put(new Integer(movementField.getName()), movementField);
		_numberOfMovementFields++;
		return movementField;
	}
	
	/**
	 * Fügt ein HashSet von <code>Node</code>-Element in die <code>_nodes</code> 
	 * Liste ein. und erhält eine inkremetelle Laufnummer, über den das Element 
	 * identifiziert werden kann. Als Rückgabe wird das HashSet mit den neuen 
	 * Laufnummern zurückgegen, damit es in den ChoiceTree eingetragen werden kann.
	 * 
	 * @param nodeList Liste von Nodes die in die _nodes List eingefügt werden
	 * @return HashSet von aktualisierten Nodes.
	 */
	public HashSet setElement(HashSet nodeList) {
		Iterator iter = nodeList.iterator();
		while(iter.hasNext()) {			
			Node node = (Node) iter.next();
			node.setName(_numberOfNodes);
			_nodeList.add(node);
			_numberOfNodes++;
		}
		return nodeList;
	}
	
	/**
	 * Zum Einfügen eines einzigen Nodes in die HashSet Liste <code>_nodes</code>.
	 * Eine inkrementelle Laufnummer wird dem Node zugeordnet. Das Node wird als
	 * HashSet zurückgegeben, da es so ohne Probleme an den ChoiceTree weiter-
	 * gegeben werden kann.
	 * 
	 * @param node Einzelnes node Element, das in <code>_nodes</code eingefügt werden soll.
	 * @return Hash Set mit dem neu hinzugefügten Node
	 */
	public HashSet setElement(Node node) {
		HashSet nodes = new HashSet();
		nodes.add(node);
		return setElement(nodes);
	}
	
	/**
	 * Liefert das HashSet mit allen Nodes zurück.
	 * 
	 * @return HashSet mit Nodes.
	 */
	public HashSet getNodes() {
		return _nodeList;
	}
	
	/**
	 * Liefert ein HashMap mit allen Movementfields zurück
	 * 
	 * @return HashMap mit allen MovementFields.
	 */
	public HashMap getMovementFieldList() {
		return _movementFieldList;
	}
	
	/**
	 * Gibt eine neues Szenario an das ValueTable.
	 * 
	 * @param valueNew Übergibt neue Basis-Informationen an das Value Table
	 */
	public void setValueNew(ValueNew valueNew) {
		_valueNew = valueNew;
	}
	
	/**
	 * Liefert die Basis-Informationen über das Szenario zurück.
	 * 
	 * @return Gibt die Basisinformationen des Szenarios zurück.
	 */
	public ValueNew getValueNew() {
		return _valueNew;
	}

	/**
	 * Liefert das momentan markierte Objekt zurück. Das Objekt kann von der 
	 * Form <code>Node</code> oder <code>MovementField</code> sein.
	 * 
	 * @return Gibt ein momentan markierte Objekt zurück. 
	 */
	public Object getMarked() {
		if(_markedMovementField != null)
			return _markedMovementField;
		else
			return _markedNode;
	}
	
	/**
	 *  Liefert den Key des Objektes zurück. Bei einem <code>Node</code> ist es 
	 * das Node Objekt selbst und bei einem <code>MovementField</code> ist es
	 * der HashMap Wert des Keys.
	 * 
	 * @return Liefert den Key des markierten Objektes zurück.
	 */
	public Object getKey() {
		return (Object) _markedKey;
	}
	
	/**
	 * Löscht das markierte Objekt aus seiner Liste. Bei einem <code>Node</code> 
	 * wird das Node Objekt gelöscht und bei einem <code>MovementField</code> 
	 * wird das MovementField gelöscht. Das gelöschte Objekt wird zurückgeliefert,
	 * damit es auch aus dem ChoiceTree gelöscht werden kann.
	 * 
	 * @param key Der Key, mit dem das Objekt identifiziert wird.
	 * @return Liefert das gelöschte Objekt zurück
	 */
	public Object remove(Object key) {
		if(_markedMovementField != null) {
			_movementFieldList.remove(key);
			return _markedMovementField;
		} else {
			_nodeList.remove(key);
			return _markedNode;
		}
	}
	
	/**
	 * Erfragt ob momentan ein markiertes Objekt existiert.
	 * 
	 * @return Liefert <code>-1</code> zurück, wenn kein markiertes Objekt 
	 * vorliegt.
	 */
	public int isMarked() {
		return _marked;
	}
	
	/**
	 * Löscht die Markierung eines Objektes
	 *
	 */
	public void unmark() {
		_marked = -1;
	}
	
	/**
	 * Übergibt die Position des Cursors. Danach wird die Liste aller <code>
	 * _movementFields</code> abgesucht, ob sich die Cursor Position innerhalb
	 * des Polygons eines MovementFields befindet. Falls sich der Cursor darin 
	 * befindet, wird ein <code>true<code> zurückgeliefert.
	 * 
	 * @param position Position des Cursors
	 * @return boolean Liefert <code>true</code> zurück wenn Cursor auf einem 
	 * MovementField steht. 
	 */
	public boolean isInMovementList(Position position) {
		boolean isInMovementField = false;
		Iterator iter = _movementFieldList.entrySet().iterator();
		while(iter.hasNext()) {
			MovementField movementField = (MovementField) ((Map.Entry) iter.next()).getValue();
			if(movementField.isPointInMovementField(position)) 
				isInMovementField = true;
		}
		return isInMovementField;
	}
	
	/**
	 * Kontrolliert ob sich unter der Cursorposition ein Beliebiges Element 
	 * befindet. Ein Element kann von der Form <code>MovementField</code> oder 
	 * <code>Node</code> sein. Wenn sich ein Objekt unter dem Cursor befindet,
	 * wird true zurückgeliefert.
	 * 
	 * @param position des Cursors
	 * @return boolean <true> wenn sich der Cursor über einem Objekt befindet.
	 */
	public boolean isElementAt(Position position) {
		_markedKey = null;
		Iterator iter = _movementFieldList.entrySet().iterator();
		while(iter.hasNext()) {
			Map.Entry me = (Map.Entry) iter.next();
			Object key =  me.getKey();
			MovementField marked = (MovementField) me.getValue();
			
			if(marked.isPointInMovementField(position)) {
				setMarked(marked, key);
			}
		}
		iter = _nodeList.iterator();
		while(iter.hasNext()) {		
			Object key = iter.next();
			Node marked = (Node) key;
			if(marked.isPointInNode(position)) {
				setMarked(marked, key);
			}
		}	
		return (_markedKey != null);
	}
	
	/**
	 * Identifiziert, ein Node oder den Knoten eines Polygons und wenn sich
	 * eines dieser Elemente an Position befindet, wird es als markiert 
	 * deklariert.   
	 * 
	 * @param position Position die Abgefragt werden soll
	 * @param zrx Zoomfaktor der X-Koordinate
	 * @param zry Zoomfaktor der Y-Koordinate
	 */
	
	public void identify(Position position, double zrx, double zry){		
		if(_markedMovementField != null) {
			int i = 0;
			ListIterator liter = _markedMovementField.getNodes().listIterator();
			while(liter.hasNext()) {
				Position posit = (Position) liter.next();
				if((posit.getX()-5/zrx)<position.getX() &&
					(posit.getX()+5/zrx)>position.getX() &&
					(posit.getY()-5/zry)<position.getY() &&
					(posit.getY()+5/zry)>position.getY()) {
						_marked = i; 
						return;
				}
				i++;
			}
			if(isElementAt(position))
					_marked = -2;
			return;
		}
		Iterator iter = _nodeList.iterator();
		while(iter.hasNext()) {
			Node node = (Node) iter.next();
			if(node.isRandomStartPoint() == true)
				continue;
			if(node.getPosition().getX()-5<position.getX() &&
					node.getPosition().getX()+5>position.getX() &&
					node.getPosition().getY()-5<position.getY() &&
					node.getPosition().getY()+5>position.getY()) {
					_marked = 0;
					setMarked(node, node);
					return;
			}
		}
		
	}
	
	/**
	 * Rückt das markierte MovementField an die angegebene Position
	 * 
	 * @param position Position an den das MovementField gerückt werden soll.
	 */
	public void movePolygonNode(Position position) {
			_markedMovementField.getNodes().set(_marked, position);
	}
	
	/**
	 * Rückt das markierte Node an die angegebene Position
	 * 
	 * @param position Position an den die das Node gerückt werden soll.
	 * @return False, falls das Node nicht in der NodeList steht.
	 */
	public boolean moveNode(Position position) {
		if(isInMovementList(position)) {
			_markedNode.setPosition(position);
			return true;
		}
		return false;
	}
	
	/**
	 * Setzt alle Markierten werte wieder zurück auf die Default-Werte.
	 */
	public void setMarkedToDefault() {
		_markedNode = null;
		_markedMovementField = null;
		_markedKey = null;
		_marked = -1;
	}
	
	/**
	 * Speichert die Position, an der die letzte Markierung vorgenommen wurde.
	 * 
	 * @param position Position der zu speichernden Markierung
	 */
	public void initialRectangle(Position position) {
		_markedPolygonNode = position;
	}
	
	/**
	 *  Liefert die gespeicherte Position der letzten Markierung zurück.
	 * 
	 * @return Position der letzten Markierung
	 */
	public Position getInitialRectangle() {
		return _markedPolygonNode;
	}
	
	/**
	 * Wenn das Rectangle positioniert wurde, wird es beim Finishing in die 
	 * <code>_movemenFields</code> eingetragen und ist damit als Rectangle 
	 * auf der Paintingarea sichtbar.
	 * Das neue Rectangle wird zrückgeliefert, damit es in den ChoiceTree
	 * eingetragen werden kann.
	 * 
	 * @param position mit der das Rectangle seine endgültige Positionierung bekommt
	 * @param zrx Zoomfaktor der X-Koordinate
	 * @param zry Zoomfaktor der Y-Koordinate
	 * @return MovementField um es an den Choicetree weitergeben zu können.
	 */
	public MovementField finishRectangle(Position position, double zrx, double zry) {
		MovementField movementField = 
			new MovementField(this.getValueNew().getSimulationTime());
		movementField.setNode(new Position(
				_markedPolygonNode.getX()/zrx,
				_markedPolygonNode.getY()/zry));
		movementField.setNode(new Position(
				_markedPolygonNode.getX()/zrx,
				position.getY()/zry));
		movementField.setNode(new Position(
				position.getX()/zrx, 
				position.getY()/zry));
		movementField.setNode(new Position(
				position.getX()/zrx,
				_markedPolygonNode.getY()/zry));
		return setElement(movementField);
	}
	
	/**
	 * Bewegt das komplette MovementField mit allen seinen Nodes an die 
	 * übergebene Position.
	 * 
	 * @param position Bewegt das markierte MovementField an die angegebene Position.
	 */
	public void moveMovementField(Position position) {
		
		if(_markedMovementField != null) {
			
			MovementField movementField = (MovementField) _movementFieldList.get(_markedKey);
			ListIterator liter = movementField.getNodes().listIterator();
			while(liter.hasNext()) {
				Position posit = (Position) liter.next();
				liter.set(new Position(posit.getX()+position.getX(),
						posit.getY()+position.getY()));
				
			}
		}
	}

	/**
	 * Diese Methode Copiert das Markierte Objekt. Da bei einer Kopie ein 
	 * komplett neues Objekt erstellt werden muss, wird vom markierte Node
	 * oder MovementField eine Tiefenkopie erstellt, da ein normal kopiertes
	 * Objekt die Referenzen zu ihrem Original beibehalten.
	 * 
	 * @return Object gibt die Kopie eines Elementes zurück, ansonsten null 
	 */
	public Object getCopy() {
		
		if(_markedMovementField != null) {
			MovementField movementField = 
				new MovementField(this.getValueNew().getSimulationTime());
			movementField.setNodes(_markedMovementField.getNodes());
			return movementField;
		} 
		if(_markedNode != null) {
			Node node = new Node(_valueNew.getSimulationTime());
			node.setPosition(_markedNode.getPosition());
			node.setRandomStartPoint(_markedNode.getRandomMovement());
			node.setRandomMovement(_markedNode.getRandomMovement());
			node.setTimeScheduler(_markedNode.getTimeScheduler());
			return node;
		}
		return null;
		
	}
	
	/**
	 * Setzt ein MovementField als markiertes Objekt ein.
	 * 
	 * @param marked Zu markierendes Objekt
	 * @param key Key des zu markierenden Objekts
	 */
	private void setMarked(MovementField marked, Object key) {
		_markedNode = null; 
		_markedMovementField = marked;
		_markedKey = key;
	}
	
	/**
	 * Setzt ein Node als markiertes Objekt ein. Da ein HashSet keinen Key 
	 * besitzt, ist hier das markierte Objekt und der Key das selbe Objekt.
	 * 
	 * @param marked Zu markierendes Objekt
	 * @param key Key des zu markierenden Objekts
	 */
	private void setMarked(Node marked, Object key) {
		_markedMovementField = null;
		_markedKey = key;
		_markedNode = marked;
	}
}