/*
 * Created on 21.08.2004
 *
 */
package mapper.DataStrukture;

/**
 * Beinhaltet alle Informationen, die benötigt werden um einen Knoten an eine andere 
 * Position zu bewegen. Anfangs und Endposition des <code>Node</code> oder <code>
 * MovementField</code>, sowie die Anfangs und Endzeit der Positionsbewegung. Sowie
 * der Speed, der verwendet werden soll. Sofern der Anfang und das Ende, sollte man sich
 * nicht die Mühe machen den Speed anzugenbe. Falls die Anfangszeit der Bewegung und
 * der Speed angegeben wird, darf man nicht den Endzeitpunkt angeben, da Speed sonst
 * überschrieben wird. <br><br>
 * 
 * _startPosition, stopPosition: Dimensionsangaben in Metern.<br>
 * _startTime, _stopTime : Die Zeit wird in Sekunden berechnet.<br>
 * _speed : wird in km/h berechnet.  <br>
 * 
 * @author Emanuel Eden
 *
 */

public class TimeSlot {
	
	// Start- und Stop-Position des Nodes oder MovementFields
	private Position _startPosition;
	private Position _stopPosition;
	// Start- und Stop-Time des Nodes oder MovementFields 
	private double _startTime;
	private double _stopTime;
	// Speed des Nodes oder des MovementFields
	private double _speed;
	
	/**
	 * Initialisierung ohne Werte
	 */
	public TimeSlot() {
		
		_startPosition = null;
		_stopPosition = null;
		_startTime = 0;
		_stopTime = 0;
		_speed = 0;
	}
	
	/**
	 * Initialisieren des TimeSlots mittels eines anderen TimeSlots
	 * 
	 * @param timeSlot TimeSlot der übergeben wird
	 */
	public TimeSlot(TimeSlot timeSlot) {
		
		_startPosition = timeSlot.getStartPosition();
		_stopPosition = timeSlot.getStopPosition();
		_startTime = timeSlot.getStartTime();
		_stopTime = timeSlot.getStopTime();
		_speed = timeSlot.getSpeed();
	}
	
	/**
	 * TimeSlot wird initialisiert mit dem Startpunkt eines Nodes oder MovementFields.
	 * 
	 * @param startPosition <code>Position</code> eines Nodes oder MovementFields 
	 */
	public TimeSlot(Position startPosition) {
		
		_startPosition = startPosition;
		_stopPosition = null;
		_startTime = 0;
		_stopTime = 0;
		_speed = 0;
	}
	
	/**
	 * TimeSlot wird initialisiert mit dem Startpunkt und der Startzeit, 
	 * 
	 * @param startPosition Position des Nodes oder MovementFields, bzw. der letzten 
	 * stopPosition
	 * @param startTime StopTime des letzten TimeSlots
	 */
	public TimeSlot(Position startPosition, double startTime) {
		
		_startPosition = startPosition;
		_stopPosition = null;
		_startTime = startTime;
		_stopTime = startTime;
		_speed = 0;
	}
	
	/**
	 * TimeSlot Erstellung mit der <code>stopTime</code> des letzten TimeSlots als 
	 * <code>startTime</code> des neuen Timeslots, sowie dem <code>speed</code> des 
	 * Nodes oder MovementFields. Die <code>stopTime</code> wird automatisch berechnet.
	 * 
	 * @param speed Geschwindigkeit des Nodes oder MovementFields in km/h
	 * @param startPosition des Nodes oder MovementFields in Metern
	 * @param stopPosition des Nodes oder MovementFields in Metern
	 * @param startTime des Nodes oder MovementFields in Sekunden
	 */
	public TimeSlot(double speed, Position startPosition, Position stopPosition, 
			double startTime) {
		
		_startPosition = startPosition;
		_stopPosition = stopPosition;
		_startTime = startTime;
		if(speed<0)
			speed = 0;
		_speed = speed;
		_stopTime = computeTime();
	}
	
	/**
	 * TimeSlot Erstellung mit der <code>stopTime</code> des letzten TimeSlots als 
	 * <code>startTime</code> des neuen Timeslots, sowie einer <code>stopTime</code>.
	 * Die berechnung des <code>speeds</code> erfolgt automatisch.
	 * 
	 * @param startTime des Nodes oder MovementFields in Sekunden
	 * @param stopTime des Nodes oder MovementFields in Sekunden
	 * @param startPosition des Nodes oder MovementFields in Metern
	 * @param stopPosition des Nodes oder MovementFields in Metern
	 */
	public TimeSlot(double startTime, double stopTime, Position startPosition, 
			Position stopPosition) {
		
		_startPosition = startPosition;
		_stopPosition = stopPosition;
		_startTime = startTime;
		if(_startTime>stopTime)
			stopTime = _startTime;
		_stopTime = stopTime;
		_speed = computeSpeed();
	}
	
	/**
	 * Setzt die Startposition des TimeSlots neu fest. Dabei wird auch der Speed neu
	 * berechnet.
	 * 
	 * @param startPosition eine neue StartPositon vom Objekt <code>Position</code>
	 */
	public void setStartPosition(Position startPosition) {
		
		_startPosition = startPosition;
		_speed = computeSpeed();
	}
	
	/**
	 * Liefert den momentanen Startpunkt des Nodes oder des MovementFields zurück
	 * 
	 * @return Position der StartPosition des TimeSlots
	 */
	public Position getStartPosition() {
		return _startPosition;
	}
	
	/**
	 * Setzt die Stopposition des TimeSlots neu fest. Dabei wird auch der Speed neu
	 * berechnet.
	 * 
	 * @param stopPosition Die StopPosition des TimeSlots
	 */
	public void setStopPosition(Position stopPosition) {
		_stopPosition = stopPosition;
	}
	
	/**
	 * Liefert den momentanen Stoppunkt des Nodes oder des MovementFields zurück
	 * 
	 * @return Position der StartPosition des TimeSlots
	 */
	public Position getStopPosition() {
		return _stopPosition;
	}
	
	/**
	 * Liefert die momentane Startzeit des Nodes oder des MovementFields zurück
	 * 
	 * @return double Liefert die Startzeit in Sekunden zurück
	 */
	public double getStartTime() {
		return _startTime;
	}
	
	/**
	 * Setzt die Startzeit des TimeSlots neu fest. Dabei wird auch der Speed neu
	 * berechnet.
	 * 
	 * @param startTime Doublewert Setzt die Startzeit des TimeSlots in Sekunden fest.
	 */
	public void setStartTime(double startTime) {
		_startTime = startTime;
		_speed = computeSpeed();
	}
	
	/**
	 * Setzt die Stopzeit des TimeSlots neu fest. Dabei wird auch der Speed neu
	 * berechnet.
	 * 
	 * @param stopTime Doublewert Setzt die Stopzeit des TimeSlots in Sekunden fest.
	 */
	public void setStopTime(double stopTime) {
		_stopTime = stopTime;
		_speed = computeSpeed();
	}
	
	/**
	 * Liefert den momentane Stopzeit des Nodes oder des MovementFields zurück
	 * 
	 * @return double Liefert die Stopzeit in Sekunden zurück
	 */
	public double getStopTime() {
		return _stopTime;
	}
	
	/**
	 * Liefert den Speed in km/h zurück.
	 * 
	 * @return double Speed in km/h
	 */
	public double getSpeed() {
		return _speed;
	}
	
	/**
	 * Schaut ob alle relevanten Werte zur Berechnung des Speeds bzw. der stopTime
	 * vorliegen.
	 * 
	 * @return boolean wert ob mittels der Werte speed und stopTime neu berechnet 
	 * werden können
	 */
	private boolean canCompute() {
		if((_startTime > -1) && ((_stopTime > -1) ||(_speed > -1))) {
			return true;
		}
		return false;
	}
	
	/**
	 * Berechnet die Entfernung von der StartPosition bis zur StopPosition in Metern.
	 * 
	 * @param startPosition Ist der Startpunkt des TimeSlots
	 * @param stopPosition Ist der Stoppunkt des TimeSlots
	 * @return liefert die Distanz zwischen den Positionen in Metern
	 */
	private double lineLength(Position startPosition, Position stopPosition) {
		double a, b;
		// Länge der Linie in Metern
		a = Math.abs(startPosition.getX() - stopPosition.getX());
		b = Math.abs(startPosition.getY() - stopPosition.getY());
		return Math.sqrt(a*a + b*b);
	}
	
	/**
	 * Berechnet den Speed mittels der Werte von Start- und StopPosition und von Start- und 
	 * Stopzeit. 
	 * 
	 * @return double Liefert den Speed in km/h zurück
	 */
	private double computeSpeed() {
		double c, time;
		c = lineLength(_startPosition, _stopPosition);
		time = _stopTime - _startTime;
		return (c*3.6)/time;
	}
	
	/**
	 * Berechnet die stopZeit des Timeslots mittels der Werte von Start- und Stop-
	 * Position und von der StartTime und dem Speed. 
	 * 
	 * @return double StopTime in Sekunden
	 */
	private double computeTime() {
		double c, time;
		c = lineLength(_startPosition, _stopPosition);
		return _startTime + ((c*3.6)/_speed);
	}
}
