/*
 * Created on 29.08.2004
 *
 */
package mapper.DataStrukture;

/**
 * Die Klasse PathLine wird von der PaintingArea verwendet, um einen dynamischen Pfad oder 
 * Linie vom momentan Aktiven MovementField oder Node zum Cursor zu generieren. Hierbei verändert
 * sich die Position source nicht, während die Cursorposition Sink ständig aktualisiert wird.
 * 
 * Teilweise wird PathLine, aber auch als Container für eine Linie verwendet.
 * 
 * @author Emanuel Eden
 *
 */
public class PathLine {
	
	// Position eines MovementFields oder eines Knotens
	Position _source;
	// Position des Cursors
	Position _sink;
	// Temporäres abspeichern der SourcePosition, da source durch den Zoomfaktor ergänzt wird.
	// _sink benötigt keine Korrektur, da sie ständig erneuert wird.
	Position _tmp;
	
	/**
	 * Initialisierung von PathLine ohne Elemente
	 *
	 */
	public PathLine() {
		_source = new Position();
		_tmp = new Position();
		_sink = new Position();
	}
	
	/**
	 * Initialisiert PathLine mit einem SourceObjekt. Ein Sourceobjekt kann entweder ein
	 * <code>MovementField</code> oder ein <code>Node</code> Objekt sein. 
	 * 
	 * @param source Position des MovementFields oder eines Nodes.
	 */
	public PathLine(Position source) {
		_source = source;
		_tmp = source;
		_sink = null;
	}
	
	/**
	 * Initialisiert PathLine mit einem SourceObjekt. Ein Sourceobjekt kann entweder ein
	 * <code>MovementField</code> oder ein <code>Node</code> Objekt sein. 
	 * Bei zrx und zry handelt es sich um den aktuellen Zoomfaktor der PaintingArea
	 * 
	 * @param source Position des MovementFields oder eines Nodes.
	 * @param zrx Zoomfaktor der X-Richtung
	 * @param zry Zoomfaktor in Y-Richtung
	 */
	public PathLine(Position source, double zrx, double zry) {
		_source = new Position(source.getX()*zrx, source.getY()*zry);
		_tmp = source;
		_sink = null;
	}
	
	/**
	 * Initialisiert PathLine mit einem SourceObjekt, sowie der aktuellen Cursor-Position. 
	 * Ein Sourceobjekt kann entweder ein <code>MovementField</code> oder ein <code>Node</code> 
	 * Objekt sein, die Cursor-Position ist ein beliebiges Koordinatenpaar in der PaintingArea
	 * 
	 * @param source Position des Source Objektes
	 * @param sink Position des Cursors
	 */
	public PathLine(Position source, Position sink) {
		_source = source;
		_tmp = source;
		_sink = sink;
	}
	
	/**
	 * Setzt eine neues Sourceobjekt ein.
	 * 
	 * @param source Position des MovementFields oder eines Nodes.
	 */
	public void setSource(Position source) {
		_source = source;
	}
	
	/**
	 * Liefert die Positionsdaten des Sourceobjektes zurück. Wird verwendet, damit die Position 
	 * in der PaintingArea gezeichnet werden kann.
	 * 
	 * @return Position Positionierungsdaten eines Nodes oder MovementFields
	 */
	public Position getSource() {
		return _source;
	}
	
	
	/**
	 * Setzt eine neu Cursorposition ein.
	 * 
	 * @param sink Cursorposition
	 */
	public void setSink(Position sink) {
		_sink = sink;
	}
	
	/**
	 * Setzt die Cursorposition mit den entsprechenden Zoomfaktoren der Paintingarea fest
	 * 
	 * @param sink Cursor-Position
	 * @param zrx Zoomfaktor der X-Richtung
	 * @param zry Zoomfaktor der Y-Richtung
	 */
	public void setSink(Position sink, double zrx, double zry) {
		_source = new Position(_tmp.getX()*zrx, _tmp.getY()*zry);
		_sink = sink;
		
	}
	
	/**
	 * Liefert die Cursorposition zurück. Wird verwendet, damit die Position in der Painting-
	 * Area gezeichnet werden kann.
	 * 
	 * @return sink Position des Cursors
	 */
	public Position getSink() {
		return _sink;
	}
}
