/*
 * Created on 14.08.2004
 *
 */
package mapper.DataStrukture;

/**
 * Die Klasse Node repräsentiert den Übertragungsknoten in einem NS2 System. Hier werden die 
 * Relevanten Daten abgespeichert, die für die Berechnung benötigt werden.
 *
 * TODO Node have to extends the <code>Position</code> Class 
 *  
 * @author Emanuel Eden
 *
 */
public class Node {
	
	// Position des Knotend
	private Position _position;
	// Beinhaltet die Bewegung des Knotens während der Simulation
	private TimeScheduler _timeScheduler = null;
	
	// Erkennungnummer im ChoiceTree un zum Aufrufen des Objekts (ist fortlaufende Nummer, 
	// die nicht beinflusst werden kann.
	private int _nodeNameNumber;
	// Gibt an, ob der Knoten einen zufallsbasierten Startpunkt bekommt. In diesem Fall ist
	// Position nicht belegt. Und wird nicht in der PaintingArea angezeigt.
	private boolean _randomStartPoint;
	// Zeigt an, ob der Knoten sich zufallsbasiert bewegt.
	private boolean _randomMovement;
	
	/**
	 * Knoten wird mit der Simulationszeit initialisiert. Die später vom TimeScheduler
	 * verwendet wird um festzustellen, wann die Simulation endet.
	 * 
	 * @param simulationTime Simulationslaufzeit
	 */
	public Node(double simulationTime) {
		
		_position = new Position();
		_randomStartPoint = false;
		_randomMovement = false;	
		_timeScheduler = new TimeScheduler(simulationTime);
	}
	
	/**
	 * Initialisierung dieses Knotens mit einem anderen Node Objekt. Dies wird zum Kopieren
	 * eines Nodes verwendet.
	 * 
	 * @param node Ein bestehender Knoten 
	 * @param simulationTime Die Simulationslaufzeit in Sekunden 
	 */
	public Node(Node node, double simulationTime) {
		_position = node.getPosition();
		_timeScheduler = node.getTimeScheduler();
		_randomStartPoint = node.isRandomStartPoint();
		_randomMovement = node.getRandomMovement();
		_timeScheduler = new TimeScheduler(simulationTime);
	}
	
	/**
	 * Initialisierung eines Knotens mit seinen Positionsangaben auf einem Movementfield.
	 * 
	 * @param x Koordinate
	 * @param y Koordinate
	 * @param simulationTime Simulationslaufzeit
	 */
	public Node(double x, double y, double simulationTime) {
		
		_position = new Position(x, y);
		_randomStartPoint = false;
		_randomMovement = false;
		_timeScheduler = new TimeScheduler(simulationTime);
	}
	
	/**
	 * Setzt einen bereits bestehenden TimeScheduler für diesen Knoten ein.
	 * 
	 * @param timeScheduler
	 */
	public void setTimeScheduler(TimeScheduler timeScheduler) {
		_timeScheduler = timeScheduler;
	}
	
	/**
	 * Liefert den TimeScheduler für diesen Knoten zurück.
	 * 
	 * @return TimeScheduler Objekt
	 */
	public TimeScheduler getTimeScheduler() {
		return _timeScheduler;
	}
	
	/**
	 * Liefert ein neues TimeSlot Element für den TimeScheduler. Sofern noch kein einziger
	 * TimeSlot im TimeScheduler besteht, wird nur die Position des MovementFields über-
	 * geben. Wenn sich jedoch bereits ein TimeSlot im TimeScheduler befindet. Wird die
	 * StopPosition des letzte TimeSlots als Anfangspunkt definiert und erhählt zudem die 
	 * zuletzt gültige Stop Zeit als Start Zeit.
	 * 
	 * @return Liefert einen neuen TimeSlot zurück
	 */
	public TimeSlot getTimeSlot() {
		if(_timeScheduler.getLastKey() == -1)
			return new TimeSlot(_position);
		else {
			return new TimeSlot(
					_timeScheduler.getLastTimeSlot().getStopPosition(), 
					_timeScheduler.getLastTimeSlot().getStopTime());
		}
	}
	
	/**
	 * Liefert die Position des letzten TimeSlots im TimeScheduler zurück. Sofern sich noch
	 * kein TimeSlot im TimeScheduler befindet wird die Position des Nodes zurückgeliefert.
	 * 
	 * @return Liefert die momentane Position des letzten TimeSlots zurück
	 */
	public Position getTimeSlotPosition() {
		if(_timeScheduler.getLastKey() == -1)
			return _position;
		else
			return _timeScheduler.getLastTimeSlot().getStopPosition();
	}
	
	/**
	 * Setzt eine neue Position für den Knoten. Wird benötigt beim Verschieben des Nodes.
	 * 
	 * @param position Neue Position
	 */
	public void setPosition(Position position) {
		_timeScheduler = new TimeScheduler(_timeScheduler.getSimTime());
		_position = position;
	}
	
	/**
	 * Liefert die momentane Position des Knotens zurück
	 * 
	 * @return Knotenposition vom Typ <code>Position</code> 
	 */
	public Position getPosition() {
		return _position;	
	}
	
	/**
	 * Setzt den Knotenpositionsstartpunkt auf einen zufallsbasierten Wert, wenn der übergebene
	 * Wert gleich <code>true</code> ist. Wenn dieser Wert <code>false</code> ist, muss 
	 * vorher oder kurz danach eine Positionierung des Knotens erfolgen, da ansonsten ein 
	 * Fehler auftreten kann. Der Knoten wird bei <code>true</code>, nicht im MovementField 
	 * angezeigt.
	 * 
	 * @param randomStartPoint
	 */
	public void setRandomStartPoint(boolean randomStartPoint) {
		if(randomStartPoint == true)
			_position = null;
		_randomStartPoint = randomStartPoint;
	}
	
	/**
	 * Liefert zurück, ob es sich bei dem Knoten um einen Zufallsbasierten Startpunkt handelt.
	 * 
	 * @return boolean <code>true</code> wenn der Startpunkt zufallsbasiert ist.
	 */
	public boolean isRandomStartPoint() {
		return _randomStartPoint;
	}
	
	/**
	 * Setzt den Knoten auf einen zufallsbasierten Bewegungsablauf. <code>true</code>, wenn 
	 * der Knoten sich zufallsbasiert bewegt.
	 * 
	 * @param randomMovement boolean
	 */
	public void setRandomMovement(boolean randomMovement) {
		_randomMovement = randomMovement;
	}
	
	/**
	 * Liefert zurück, ob sich der Knoten zufallsbasiert bewegt, dann ist der Rückgabewert 
	 * <code>true</code> ansonsten ist er <code>false</code>
	 * 
	 * @return boolen
	 */
	public boolean getRandomMovement() {
		return _randomMovement;
	}
	
	/**
	 * Setzt den Internen NodeNamen fest, der als forlaufende Integernummer deklariert wirde. 
	 * Der Wert wird intern gesetzt und kann im Verlauf des Programms nicht eigenständig definiert
	 * werden.
	 * 
	 * @param nameNumber fortlaufende Integernummer.
	 */
	public void setName(int nameNumber) {
		_nodeNameNumber = nameNumber;
	}
	
	/**
	 * Liefert den Namen des Knotens zurück. Wird verwendet um den Knoten im Choicetree wieder-
	 * zuerkennen.
	 * 
	 * @return int Liefert den NamenNummer des Knotens zurück.
	 */
	public int getName() {
		return _nodeNameNumber;
	}
	
	/**
	 * Diese Methode wird verwendet, um sagen zu können, ob sich der Cursor gerade über dem 
	 * Knoten befindet. Diese Methode wird zum Markieren des Knotens auf der Paintingarea
	 * verwendet. Hierbei muss sich der Cursor ca. im Radius von 5 Pixeln um den Knoten befinden.
	 * 
	 * @param cursorPosition Liefert die Position des Cursors an.
	 * @return boolen <code>true</code> falls sich der Cursor in der nähe des Nodes befindet.
	 */
	public boolean isPointInNode(Position cursorPosition) {
		int x = (int) (cursorPosition.getX() - _position.getX()); 
		int y = (int) (cursorPosition.getY() - _position.getY());
		if(x > -5 && x < 5 && y > -5 && y < 5)
			return true;
		return false;
	}
}
