/*
 * Created on 14.08.2004
 *
 */
package mapper.DataStrukture;

import java.util.*;
import java.awt.Polygon;
/**
 * Die Klasse MovementField beinhaltet in erster Line die Knoten des Polygons, in denen 
 * sich später die Nodes für das NS2-System bewegen können. Hierbei werden die Knoten 
 * doppelt verwaltet, einmal als double und einmal als Polygon. Diese Verfahrensweise 
 * wird verwendet, da die double Werte bessere und genauere Ergebnisse, gerade beim 
 * Skalieren des Zooms, ergeben. Währende die Polygone nur für die Darstellung des 
 * Polygons selbst gelten. Die "workbench" (PaintingArea) rechnet nur mit Integerwerten 
 * Gerade die Berechnung, ob sich ein gegebener Knoten in einem Poligon befindet wird so 
 * teilweise minimiert, da die Polygonpunkte erste von der ArrayList ins Poligon überführt
 * werden müssten.
 * Eine andere wesentliche Variable ist _movementFieldNameNumber, da das MovementField keinen
 * speziellen Namen, sondern nur eine Nummerierung erhält.
 * 
 * @author Emanuel Eden
 *
 */
public class MovementField {
	
	// Maximalwert einer double Variable
	final static double MAXVALUE = 1.7e308;
	// Minimalwert einer double Variable
	final static double MINVALUE = 1.7e-308;
	
	// Beinhaltet die PolygonKnoten eines MovementFields
	private ArrayList _polygonNodes = new ArrayList();
	//private Position _tmp = new Position();
	// Speichert die Bewegungen des MovementFields
	private TimeScheduler _timeScheduler = null;
	
	// Die Laufnummer des MovementField, ist nur zum wiedererkennen im ChoiceTree
	private int _movementFieldNameNumber;
	// Gibt an ob das MovementField ein bewegliches Objekt ist 
	private boolean _isStatic = true;
	
	/**
	 * Objekt wird mit der Simulationszeit initialisiert. Diese Zeit benötigt der TimeScheduler
	 * um festzustellen, ob die Simulationszeit überschritten wird 
	 *
	 */
	public MovementField(double simulationTime) {
		
		_polygonNodes = new ArrayList();
		_isStatic = true;
		_timeScheduler = new TimeScheduler(simulationTime);
	}
	
	/**
	 * Dieses MovementField mit einem fremden MovementField initilalisieren und der 
	 * Simulationszeit.
	 * 
	 * @param movementField bestehendes MovementField
	 */
	public MovementField(MovementField movementField) {
		
		_polygonNodes = movementField.getNodes();
		_movementFieldNameNumber = movementField.getName();
		_isStatic = movementField.getStaticStatus();
		_timeScheduler = movementField.getTimeScheduler();
	}
	
	/**
	 * Konstruktor erhählt ein Polygon in Form einer ArrayList. Die ArrayList gilt als
	 * Grundlage für die Berechnungen des MovementFields. Dazu die Simulationszeit.
	 * 
	 * @param polygon ArrayList mit Nodes oder Punkten, der Form <code>Position</code>
	 */
	public MovementField(ArrayList polygon, double simulationTime) {
		
		_polygonNodes = new ArrayList(polygon);
		_timeScheduler = new TimeScheduler(simulationTime);
	}
	
	/**
	 * Setzt den Namen des _movementFields fest. Hierbei ist der Name zum Erkennen des 
	 * MovementFields nur ein Integerwert, der bei jedem neuen MovementField incementiert 
	 * wird.
	 * 
	 * @param nameNumber ist ein Integerwert, den das MovementField erhält
	 */
	public void setName(int nameNumber) {
		_movementFieldNameNumber = nameNumber;
	}
	
	/**
	 * Liefert den Namen des MovementFields zurück.
	 * 
	 * @return int Name des MovementFields.
	 */
	public int getName() {
		return _movementFieldNameNumber;
	}
	
	/**
	 * Konvertiert eine Liste von Knoten in einer Array list zu einem Polygon Objekt.
	 * 
	 * @param polygon
	 */
	public void setNodes(ArrayList polygon) {
		
		_polygonNodes = new ArrayList(polygon);
	}
	/**
	 * Ergänzt das MovementField um einen Knoten.
	 * 
	 * @param node Von der Form <code>Position</code>
	 */
	public void setNode(Position node) {
		_polygonNodes.add(node);
	}
	
	/**
	 * Liefert eine ArrayList allen Knoten zurück, die das MovementField benennen.
	 * 
	 * @return ArrayList von Knoten als <code>Position</code> Objekt.
	 */
	public ArrayList getNodes() {
		return _polygonNodes;
	}
	
	/**
	 * Diese Methode liefert den Duchschnittswert aller MovementField Polygone, die 
	 * in _polygonNodes abgespeichert sind. Die Methode wird benötigt, damit beim 
	 * Erstellen des Pfades, nur eine einzige Positionsangabe für die Berechnung 
	 * der Bewegung existiert. 
	 * 
	 * @return Position Mittelwert aller Polygonepositionen
	 */
	public Position getPosition() {
		
		double xMin = MAXVALUE;
		double yMin = MAXVALUE;
		double xMax = MINVALUE;
		double yMax = MINVALUE;
		
		Iterator iter = _polygonNodes.iterator();
		while(iter.hasNext()) {
			Position position = (Position) iter.next();
			if(position.getX() < xMin)
				xMin = position.getX();
			if(position.getY() < yMin)
				yMin = position.getY();
			if(position.getX() > xMax)
				xMax = position.getX();
			if(position.getY() > yMax)
				yMax = position.getY();
		}
		return new Position(xMin+(xMax-xMin)/2, yMin+(yMax-yMin)/2);
	}
	
	/**
	 * Deklariert das MovementField als Statisch. Sofern <code>isStatic</code> gleich 
	 * <code>true</code> ist, zählt dieses MovementField nicht zu den beweglichen Objekten.
	 * Andernfalls kann der kompletten Map ein Vektor zugewiesen werden, der alle 
	 * MovementFields, die mit <code>isStatic</code> gleich <code>false</code> sind, in 
	 * diese Richtung bewegen werden können.  
	 * 
	 * @param isStatic Boolean ob <code>true</code> oder <code>false</code>
	 */
	public void setStaticStatus(boolean isStatic) {
		_isStatic = isStatic;
	}
	
	/**
	 * Gibt zurück, ob das Movementfield statisch oder nicht statisch ist. 
	 * Für eine genauere Beschreibung, was statisch bedeutet, siehe die Beschreibung von
	 * <code>setStaticStatus</code>
	 * 
	 * @return Liefert boolean Wert true oder False zurück, wenn das MovementField statisch ist. 
	 */
	public boolean getStaticStatus() {
		return _isStatic;
	}

	/**
	 * Gibt aus, ob sich der Angegebene Knoten vom Objekt <code>Position</code> innerhalb 
	 * <code>true</code> oder auserhalb <code>false</code> des Polygons befindet.
	 * 
	 * @param node vom Objekt <code>Position</code>
	 * @return boolean Wert.
	 */
	public boolean isPointInMovementField(Position node) {
		return getPolygon().contains(node.getX(), node.getY());
	}
	
	/**
	 * Liefert das das MovementField als Polygon zurück. Durch das Polygon kann sehr leicht
	 * berechnet werden, ob sich ein Punkt im Polygon befindet. 
	 * 
	 * @return Polygon liefert das MovementField als Polygon Objekt zurück
	 */
	public Polygon getPolygon() {
		Polygon polygon = new Polygon();
		Iterator iter = _polygonNodes.iterator();
		while(iter.hasNext()) {
			Position node = (Position) iter.next();
			polygon.addPoint((int) node.getX(), (int) node.getY());
		}
		return polygon;
	}

	/**
	 * Setzt einen neuen TimeScheduler in das MovementField ein. Dies ist nötig, da alle 
	 * MovementFields nur die selbe Bwegenung machen dürfen. Nachdem die Bewegung des ersten
	 * MovementFields gesetzt wurde, werden alle weiter deklarierten MovementFields mit dem
	 * ersten TimeScheduler versehen.
	 *  
	 * @param timeScheduler erhält einen bestehenden TimeScheduler
	 */
	public void setTimeScheduler(TimeScheduler timeScheduler) {
		_timeScheduler = timeScheduler;
	}
	
	/**
	 * Liefert den TimeScheduler des MovementFields
	 * 
	 * @return TimeScheduler liefert den momentanen TimeScheduler
	 */
	public TimeScheduler getTimeScheduler() {
		return _timeScheduler;
	}
	
	/**
	 * Liefert ein neues TimeSlot Element für den TimeScheduler. Sofern noch kein einziger
	 * TimeSlot im TimeScheduler besteht, wird nur die Position des MovementFields über-
	 * geben. Wenn sich jedoch bereits ein TimeSlot im TimeScheduler befindet. Wird die
	 * StopPosition des letzte TimeSlots als Anfangspunkt definiert und erhählt zudem die zu
	 * letzt gültige StopTime als StartTime.
	 *  
	 * @return TimeSlot Liefert den letzten TimeSlot zurück. 
	 */
	public TimeSlot getTimeSlot() {
		if(_timeScheduler.getLastKey() == -1)
			return new TimeSlot(getPosition());
		else 
			return new TimeSlot(
					_timeScheduler.getLastTimeSlot().getStopPosition(),
					_timeScheduler.getLastTimeSlot().getStopTime());
	}
	
	/**
	 * Liefert die Position des letzten TimeSlots im TimeScheduler zurück. Sofern sich noch
	 * kein TimeSlot im TimeScheduler befindet wird die Position des MovementField zurück-
	 * geliefert.
	 * 
	 * @return Position des letzten gültigen TimeSlots oder der MovementField Position
	 */
	public Position getTimeSlotPosition() {
		if(_timeScheduler.getLastKey() == -1)
			return getPosition();
		else
			return _timeScheduler.getLastTimeSlot().getStopPosition();
	}
	
}
