/***************************************************************************
                          RTSPClientLayer.cc
                          -------------------
    begin                : Sun Jun 09 2002
  ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*! \file RTSPClientLayer.cc
This file contains the implementation of member functions of the class RTSPClientLayer.
*/
					
#include "omnetpp.h"
#include "SocketInterfacePacket.h"
#include "ip_address.h"
#include "RTSPClientLayer.h"
#include "RTSPMessage_m.h"
#include "RTPInterfacePacket.h"

Define_Module(RTSPClientLayer);

//
// methods inherited from cSimpleModule
//


void RTSPClientLayer::initialize() {
	
	_timeout = par("timeout");
	_portRTP = IN_Port((int)(par("portNumberRTSP").longValue()));
	_socketFd = Socket::FILEDESC_UNDEF;
	_debug = par("debug");
	_cseq = 0;
	setPhase("INIT");
	
};



void RTSPClientLayer::handleMessage(cMessage *msg) {

  	if (msg->arrivalGateId() == findGate("fromSocketLayer")) {
		handleMessageFromSocketLayer(msg);
	}
	else if (msg->arrivalGateId() == findGate("fromApp")) {
		handleMessageFromApp(msg);
	}
	else if (msg->arrivalGateId() == findGate("fromRTP")) {
		handleMessageFromRTP(msg);
	}
	else {
		handleSelfMessage(msg);
	}

};



void RTSPClientLayer::handleMessageFromSocketLayer(cMessage *msg) {

	SocketInterfacePacket *sifpIn = (SocketInterfacePacket *)msg;

	if (sifpIn->action() == SocketInterfacePacket::SA_SOCKET_RET) {
		_socketFd = sifpIn->filedesc();
		delete sifpIn;

		SocketInterfacePacket *sifpOut = new SocketInterfacePacket("connect()");
		sifpOut->connect(_socketFd, IN_Addr(request->getDestinationAddress()), _portRTP);

		send(sifpOut, "toSocketLayer");
	}

	else if (sifpIn->action() == SocketInterfacePacket::SA_CONNECT_RET) {
		delete sifpIn;
		
		request->setRtspVersion(1.0);
		_cseq++;
		request->setCseq(_cseq);				
		IPAddress ipaddr(request->getTransport());
	
		// create an RTPInterfacePacket to enter the session
		RTPInterfacePacket *rifpOut = new RTPInterfacePacket("startSession()");
		rifpOut->enterSession(opp_strdup("commonName"), opp_strdup(par("profileName")), par("bandwidth"), request->getTransport(), IN_Port((int)(par("portNumberRTP").longValue())));
	
		// and send it to the rtp layer
		send(rifpOut, "toRTP");
	

		if (ipaddr.isMulticast()) {
			char *transport = (char *)malloc(60);
			sprintf(transport, "RTP;multicast;destination=%s;port=%d", request->getTransport(), (int)(par("portNumberRTP").longValue()));
			request->setTransport(transport);
		}
		else {
			char *transport = (char *)malloc(60);
			sprintf(transport, "RTP;unicast;destination=%s;client_port=%d", request->getTransport(), (int)(par("portNumberRTP").longValue()));
			request->setTransport(transport);
		}   
		if (_debug) {
			writeRequestContents(request);
		}	   
	
		SocketInterfacePacket *sifpOut = new SocketInterfacePacket("rtspRequest()");	

		requestCopy = (rtspRequest *)request->dup();		

		if (!opp_strcmp(par("rtspTransportProtocol"), "UDP")) {
				
			timeoutMsg = new cMessage("timeout");
			scheduleAt(simTime()+_timeout, timeoutMsg);
		}
		 
		sifpOut->write(_socketFd, request);
		send(sifpOut, "toSocketLayer");
				
	}

	else if (sifpIn->action() == SocketInterfacePacket::SA_READ_RET) {
		
		if (timeoutMsg->isScheduled()) {
			delete cancelEvent(timeoutMsg);
		}
			
		handleResponse(sifpIn);
	}

	else {
		ev << "RTSPClientLayer: Unknown SocketInterfacePacket type " << sifpIn->action() << " !" << endl;
	}

};


void RTSPClientLayer::handleResponse(SocketInterfacePacket *sifp) {
	
	rtspResponse *response = (rtspResponse *)sifp->decapsulate();	
	delete sifp;
	
	FSM_Switch(fsm) {
		case FSM_Exit(INIT):
			handleResponseStateInit(response);
		break;
		
		case FSM_Enter(READY):
		break;
			
		case FSM_Exit(READY):
			handleResponseStateReady(response);
		break;

		case FSM_Enter(PLAYING):
		break;
	 
		case FSM_Exit(PLAYING):
			handleResponseStatePlaying(response);
		break;
	}		

};



void RTSPClientLayer::handleResponseStateInit(rtspResponse *response) {

	_sessionId = response->getSessionId();

	if (_debug) {
		writeResponseContents(response);
	}
	
	_cseq = response->getCseq();
	delete response;
	
	if (requestCopy->getMethod() == TEARDOWN) {
		closeSession();
	}
	
	delete requestCopy;
	
	setPhase("READY");
	FSM_Goto(fsm, READY);

};



void RTSPClientLayer::handleResponseStateReady(rtspResponse *response) {
	
	if (_debug) {
		writeResponseContents(response);	
	}		
	
	if (response->getStatusCode() >= 500) {
		opp_error("Server Error - The server failed to fulfill an apparently valid request. ReasonPhrase: %s. StatusCode: %d.", response->getReasonPhrase(), response->getStatusCode());
	}	
	else if (response->getStatusCode() >= 400) {
		opp_error("Client Error - The request contains bad syntax or cannot be fullfilled. ReasonPhrase: %s. StatusCode: %d.", response->getReasonPhrase(), response->getStatusCode());
	}	

	delete response;
	
	if (requestCopy->getMethod() == TEARDOWN) {
		closeSession();
	}
        else {	
        	delete requestCopy;
	}
	setPhase("PLAYING");
	FSM_Goto(fsm, PLAYING);

};



void RTSPClientLayer::handleResponseStatePlaying(rtspResponse *response) {

	if (_debug) {
		writeResponseContents(response);
	}
	
	if (response->getStatusCode() >= 500) {
		opp_error("Server Error - The server failed to fulfill an apparently valid request. ReasonPhrase: %s. StatusCode: %d.", response->getReasonPhrase(), response->getStatusCode());
	}	
	else if (response->getStatusCode() >= 400) {
		opp_error("Client Error - The request contains bad syntax or cannot be fullfilled. ReasonPhrase: %s. StatusCode: %d.", response->getReasonPhrase(), response->getStatusCode());
	}	
	
	delete response;

	if (requestCopy->getMethod() == TEARDOWN) {
		closeSession();
	}
	
	delete requestCopy;

	setPhase("READY");			
	FSM_Goto(fsm, READY);

};



void RTSPClientLayer::handleSelfMessage(cMessage *msg) {  

	if (!opp_strcmp(msg->className(), "cMessage")) {
		SocketInterfacePacket *sifpOut = new SocketInterfacePacket("rtspRequest()");
		
		sifpOut->write(_socketFd, requestCopy);
		if (_debug) {
			ev << "RTSPClientLayer: Timeout!" << endl;
			writeRequestContents(requestCopy);
		}
		send(sifpOut, "toSocketLayer");

		
		requestCopy = (rtspRequest *)requestCopy->dup();		
		timeoutMsg = new cMessage("timeout");
		scheduleAt(simTime()+_timeout, timeoutMsg);
	}

	else {
		opp_error("RTSPClientLayer: Unknown Selfmessage");
	}
	
	delete msg;

};



void RTSPClientLayer::closeSession() {  
	
	SocketInterfacePacket *sifpOut = new SocketInterfacePacket("closeSocket()");
						
	sifpOut->close(_socketFd);
	send(sifpOut, "toSocketLayer");

	cMessage *msg = new cMessage("closeSession()");
	cout << "Closing the session" << endl;
	// sending the msg with lot of delay for testing!
	sendDirect(msg, 50, parentModule()->parentModule(), "fromClientProcess");
	delete requestCopy;
	
};



void RTSPClientLayer::handleMessageFromApp(cMessage *msg) {  

	request = (rtspRequest *)msg;		
	
	if (_socketFd == Socket::FILEDESC_UNDEF) {
		SocketInterfacePacket *sifp = new SocketInterfacePacket("socket()");
		sifp->socket(Socket::AF_INET, Socket::SOCK_DGRAM, Socket::UDP);
		send(sifp, "toSocketLayer");
	}
	else {

								
		request->setRtspVersion(1.0);
		request->setSessionId(_sessionId);				
		_cseq++;
		request->setCseq(_cseq);				

		if (_debug) {
			writeRequestContents(request);
		}	   
		
		if (request->getMethod() == TEARDOWN) {
			RTPInterfacePacket *rifpOut = new RTPInterfacePacket("leaveSession()");
			rifpOut->leaveSession();
			send(rifpOut, "toRTP");
		}	
		requestCopy = (rtspRequest *)request->dup();	
	
		if (!opp_strcmp(par("rtspTransportProtocol"), "UDP")) {
				
			timeoutMsg = new cMessage("timeout");
			scheduleAt(simTime()+_timeout, timeoutMsg);
		}
			 
		SocketInterfacePacket *sifpOut = new SocketInterfacePacket("rtspRequest()");	
		sifpOut->write(_socketFd, request);
		send(sifpOut, "toSocketLayer");
	}

};



void RTSPClientLayer::handleMessageFromRTP(cMessage *msg) {  
	RTPInterfacePacket *rinpIn = (RTPInterfacePacket *)msg;
	
	if (rinpIn->type() == RTPInterfacePacket::RTP_IFP_SESSION_ENTERED) {
	}
	
	else if (rinpIn->type() == RTPInterfacePacket::RTP_IFP_SESSION_LEFT) {
	}	

	delete rinpIn;

};



void RTSPClientLayer::writeRequestContents(rtspRequest *request) {
	
	ev << "\nRTSPClientLayer ProcessID "<< id () << ": sending request...\n"
	   << "-----------------------------------------------------------------------------\n"; 
		switch (request->getMethod()) {
		case SETUP: 
				ev << "SETUP"
				<< " rtsp://" 
			   	<< request->getDestinationAddress() 
				<< "/" 
				<< request->getFileName() 
				<< " RTSP/"
				<< request->getRtspVersion() 
				<< endl
				<< "CSeq: " 
				<< request->getCseq() 
				<< endl
				<< "Transport: " 
				<< request->getTransport() 
				<< endl;	
		break;
		
		case PLAY:
				ev << "PLAY"
				<< " rtsp://" 
			   	<< request->getDestinationAddress() 
				<< "/" 
				<< request->getFileName() 
				<< " RTSP/"
				<< request->getRtspVersion() 
				<< endl
				<< "CSeq: " 
				<< request->getCseq() 
				<< endl
				<< "Session: " 
				<< request->getSessionId() 
				<< endl	
				<< "Range: " 
				<< request->getRange() 
				<< endl;	

		break;
		
		case PAUSE: 
				ev << "PAUSE"
				<< " rtsp://" 
			   	<< request->getDestinationAddress() 
				<< "/" 
				<< request->getFileName() 
				<< " RTSP/"
				<< request->getRtspVersion() 
				<< endl
				<< "CSeq: " 
				<< request->getCseq() 
				<< endl
				<< "Session: " 
				<< request->getSessionId() 
				<< endl;	
		break;
		
		case TEARDOWN:
			ev << "TEARDOWN"
				<< " rtsp://" 
			   	<< request->getDestinationAddress() 
				<< "/" 
				<< request->getFileName() 
				<< " RTSP/"
				<< request->getRtspVersion() 
				<< endl
				<< "CSeq: " 
				<< request->getCseq() 
				<< endl
				<< "Session: " 
				<< request->getSessionId() 
				<< endl;	
		}			
	
	ev << "-----------------------------------------------------------------------------\n\n";		

};



void RTSPClientLayer::writeResponseContents(rtspResponse *response) {

	ev << "\nRTSPClientLayer ProcessID "<< id () << ": receiving response...\n"
	   << "-----------------------------------------------------------------------------\n"
	   << "RTSP/" 
	   << response->getRtspVersion() 
	   << " " 
	   << response->getStatusCode() 
	   << " " 
	   << response->getReasonPhrase()
	   << endl
	   << "CSeq: " 
	   << response->getCseq() 
	   << endl
	   << "Session: " 
	   << response->getSessionId() 
	   << endl;
	if (opp_strcmp(response->getRtpInfo(), " ")) {
	ev << "RTP-Info: " 
	   << response->getRtpInfo()
	   << endl;
	}   
	ev << "-----------------------------------------------------------------------------\n\n"; 

};
