/************************************************************************
 ** This file is part of the network simulator Shawn.                  **
 ** Copyright (C) 2004-2007 by the SwarmNet (www.swarmnet.de) project  **
 ** Shawn is free software; you can redistribute it and/or modify it   **
 ** under the terms of the BSD License. Refer to the shawn-licence.txt **
 ** file in the root of the Shawn source tree for further details.     **
 ************************************************************************/
#include "_legacyapps_enable_cmake.h"
#ifdef ENABLE_WS08

#include "legacyapps/ws08/ws08_processor.h"
#include "legacyapps/ws08/ws08_message.h"
#include "sys/taggings/basic_tags.h"
#include "sys/simulation/simulation_controller.h"
#include "sys/simulation/simulation_environment.h"
#include "sys/node_movements/linear_movement.h"
#include "sys/node.h"
#include "apps/reading/reading_keeper.h"
#include "sys/misc/random/random_variable.h"
#include "sys/misc/random/random_variable_keeper.h"
#include "sys/node_movements/linear_movement.h"
#include <iostream>
#include <limits>
#include <cmath>

namespace ws08
{
   const int Ws08Processor::UNKNOWN_DIST = std::numeric_limits<int>::max();
   // ----------------------------------------------------------------------
   Ws08Processor::
   Ws08Processor()
      : connected_      ( false ),
        gateway_        ( false ),
        hop_dist_       ( UNKNOWN_DIST ),
        send_period_    ( 6 ),
        predecessor_    ( NULL ),
        max_outlier_frac_   ( 0.5 ),
        outlier_difference_ ( 20 ),
        sensor_malfunction_ ( false )
   {}
   // ----------------------------------------------------------------------
   Ws08Processor::
   ~Ws08Processor()
   {}
   // ----------------------------------------------------------------------
   void
   Ws08Processor::
   special_boot( void )
      throw()
   {
      gateway_ = true;
      connected_ = true;
      hop_dist_ = 0;
      
      // gateway sends initial flooding message in the beginning, but only 
      // if the shortestpath-task has not already set a tag
      if ( owner().find_tag( "hop_count" ) == NULL )
      {
         send( new Ws08FloodingMessage( 1 ) );
      }
   }
   // ----------------------------------------------------------------------
   void
   Ws08Processor::
   boot( void )
      throw()
   {
      const shawn::SimulationEnvironment& se = owner().world().
                                    simulation_controller().environment();
      if ( gateway_ )
      {
         send_period_ = se.optional_int_param( "flood_period", 
                                                send_period_ );
      }
      else 
      {
         init_sensor();
         
         // because we know that the temperature reading returns values between -20 and 20, we
         // directly transform the temperature
         owner_w().write_simple_tag<double>( "temperature", sensor_value() / 40 + 0.5 );
         
         send( new Ws08LocalTemperatureMessage( sensor_value() ) );
         send_period_ = se.optional_int_param( "state_period", 
                                                send_period_ );
         max_outlier_frac_ = se.optional_double_param( "max_outlier_frac", 
                                                       max_outlier_frac_ );
         outlier_difference_ = se.optional_double_param( "outlier_diff", 
                                                         outlier_difference_ );
         
         // if hop_count is already set (and thus there should also be a predecessor
         // available) by the shortestpath-task, then read in these values directly.
         if ( owner().find_tag( "hop_count" ) != NULL )
         {
            hop_dist_ = owner().read_simple_tag<int>( "hop_count" );
            std::string pred = owner().read_simple_tag<std::string>( "predecessor" );
            predecessor_ = owner().world().find_node_by_label( pred );
            connected_ = true;  
         }
      }
      
      int round = owner().world().simulation_round();
      event_handle_ = owner_w().world_w().scheduler_w().new_event( 
                                  *this, round + send_period_, NULL );
   }
   // ----------------------------------------------------------------------
   bool
   Ws08Processor::
   process_message( const shawn::ConstMessageHandle& mh ) 
      throw()
   {  
      const Ws08FloodingMessage* floodmsg = dynamic_cast<const Ws08FloodingMessage*> ( mh.get() );
      if ( floodmsg != NULL && owner() != floodmsg->source() )
		{
         handle_flooding_message_node( *floodmsg );
			return true;
		}
      
      const Ws08RouteTemperatureMessage* temproutemsg = dynamic_cast<const Ws08RouteTemperatureMessage*> ( mh.get() );
      if ( temproutemsg != NULL && owner() == temproutemsg->destination() )
      {
         if ( gateway_ ) 
            handle_temperature_message_gate( *temproutemsg );
         else 
            handle_temperature_message_node( *temproutemsg );
               
         return true;
      }
      
      const Ws08LocalTemperatureMessage* tempmsg = dynamic_cast<const Ws08LocalTemperatureMessage*> ( mh.get() );
      if ( tempmsg != NULL && owner() != tempmsg->source() )
      {
         if (!gateway_)
            handle_local_temperature_message_node( *tempmsg );
         
         return true;
      }
      
      return shawn::Processor::process_message( mh );
   }
   // ----------------------------------------------------------------------
   void
   Ws08Processor::
   work( void )
      throw()
   {
      if ( gateway_ )
      {
         if ( owner().world().simulation_round() == 5 )
            init_movement();
         
         double min_temp = std::numeric_limits<double>::max(); 
         double max_temp = std::numeric_limits<double>::min();
         double avg_temp = 0;
                  
         for ( std::map<const shawn::Node*, double>::iterator
                  it = temperatures_.begin();
                  it != temperatures_.end();
                  ++it )
         {
            if ( it->second < min_temp ) min_temp = it->second;
            if ( it->second > max_temp ) max_temp = it->second;
            avg_temp += it->second;
         }
         avg_temp /= (double)temperatures_.size();
                  
         INFO( logger(), "Number of temperature-nodes known to gate:  " << temperatures_.size() );
         INFO( logger(), "  Min temperature:  " << min_temp );
         INFO( logger(), "  Max temperature:  " << max_temp );
         INFO( logger(), "  Avg temperature:  " << avg_temp );
      }
   }
   // ----------------------------------------------------------------------
   void 
   Ws08Processor::
   timeout( shawn::EventScheduler&, shawn::EventScheduler::EventHandle, 
            double, shawn::EventScheduler::EventTagHandle& ) 
      throw()
   {
      if ( !gateway_ && predecessor_ )
      {
         start_temperature_matching();
      }
      
      int round = owner().world().simulation_round();
      event_handle_ = owner_w().world_w().scheduler_w().new_event( 
                                             *this, round + send_period_, NULL );
   }
   // ----------------------------------------------------------------------
   void 
   Ws08Processor::
   handle_flooding_message_node( const Ws08FloodingMessage& flooding ) 
      throw()
   {   
      if ( hop_dist_ <= flooding.hops() )
         return;
      
      hop_dist_ = flooding.hops();
      predecessor_ = &flooding.source();
      
      owner_w().write_simple_tag<int>( "hop_count", hop_dist_ );
      owner_w().write_simple_tag<std::string>( "predecessor", predecessor_->label() );
            
      // forward the flooding message 
      send( new Ws08FloodingMessage( hop_dist_ + 1 ) );
                  
      // if not connected, set to connected and start new timer for sending
      // state messages
      if ( !connected_ )
      {
         connected_ = true;
         int round = owner().world().simulation_round();
         event_handle_ = owner_w().world_w().scheduler_w().new_event( 
                                       *this, round + send_period_, NULL );
      }
      
      //INFO( logger(), owner().id() 
      //                  << " stores " << flooding.source().id()
      //                  << " as new predecessor (" 
      //                  << flooding.hops() << " hops)." );
   }
   // ----------------------------------------------------------------------
   void 
   Ws08Processor::
   handle_temperature_message_node( const Ws08RouteTemperatureMessage& tempmsg )
      throw()
   {
      // only connected nodes are allowed to forward messages (so predecessor
      // must not be 0, of course). moreover, to suppress circles in routing, 
      // if thre topology changes over time, only messages from nodes with a
      // greater hop-distance to the gateway are accepted. to simplify
      // matters, there is no information sent back to state.source(), and the
      // message is just dropped
      if ( predecessor_ && tempmsg.msg_source_hops() > hop_dist_ )
      {
         send( new Ws08RouteTemperatureMessage( tempmsg.initiator(),
                                                *predecessor_, 
                                                tempmsg.temperature(),
                                                hop_dist_) );
      }
   }
   // ----------------------------------------------------------------------
   void 
   Ws08Processor::
   handle_temperature_message_gate( const Ws08RouteTemperatureMessage& tempmsg ) 
      throw()
   {
      temperatures_[ &tempmsg.initiator() ] = tempmsg.temperature();
            
      //INFO( logger(), "Received temperature message from " 
      //         << state.initiator().id() 
      //         << " with " << state.temperature() << "." );
   }
   // ----------------------------------------------------------------------
   void 
   Ws08Processor::
   handle_local_temperature_message_node( const Ws08LocalTemperatureMessage& tempmsg )
      throw()
   {
      temperatures_[ &tempmsg.source() ] = tempmsg.temperature();
   }
   // ----------------------------------------------------------------------
   void
   Ws08Processor::
   start_temperature_matching( void ) 
      throw()
   {
      double temp = sensor_value();
      send( new Ws08LocalTemperatureMessage( temp ) );
      
      if ( temperatures_.empty() )
         return;
      
      int outliers = 0;
      for ( const_temperature_map_iterator
               it = temperatures_.begin();
               it != temperatures_.end();
               ++it )
      {
         if ( std::abs(temp - it->second) >= outlier_difference_ )
            outliers++;
      }
      
      double outlier_frac = outliers / (double)temperatures_.size();
      if ( outlier_frac < max_outlier_frac_ )
      {
         send( new Ws08RouteTemperatureMessage( owner(), *predecessor_, temp, hop_dist_ ) );
      }
   }
   // ----------------------------------------------------------------------
   void
   Ws08Processor::
   init_sensor( void )
      throw()
   {
      reading::ReadingKeeper& rk = owner_w().world_w().simulation_controller_w().reading_keeper_w();
      reading::ReadingHandle reading = rk.find_w( "ws08_temperature" );
      if ( !reading.is_null() )
      {
         temperature_reading_ = dynamic_cast<reading::DoubleReading*>( reading.get() );
         double err_malfunction_frac = owner().world().simulation_controller().
            environment().optional_double_param( 
                                                "malfunction_frac", 0.0 );
         shawn::ConstRandomVariableHandle uni_rnd = 
            owner().world().simulation_controller().
                             random_variable_keeper().find( "uni[0;1]" );
      
         if ( err_malfunction_frac > *uni_rnd )
            sensor_malfunction_ = true;
      }
      else
         WARN( logger(), "Reading 'ws08_temperature' not registered." );
   }
   // ----------------------------------------------------------------------
   double 
   Ws08Processor::
   sensor_value( void ) 
      throw()
   {
      if ( temperature_reading_.is_not_null() ) 
         return temperature_reading_->value( owner().real_position() );
      
      WARN( logger(), "No temperature reading available." );
      return 0.0; 
   }
   // ----------------------------------------------------------------------
   void
   Ws08Processor::
   init_movement( void )
      throw()
   {
      double width  = owner().world().upper_right().x() 
         - owner().world().lower_left().x();
      double height = owner().world().upper_right().y() 
         - owner().world().lower_left().y();
         
      shawn::Vec new_pos = shawn::Vec(
            owner().world().lower_left().x() + width / 2,
            owner().world().lower_left().y() + height / 2,
            0.0 );
         
      shawn::LinearMovement *lm = new shawn::LinearMovement();
      lm->set_parameters( 
            ((new_pos - owner().real_position()) / 20.0).euclidean_norm(), 
            new_pos, 
            owner_w().world_w() );
      owner_w().set_movement( lm );
   }   
   
   
   
}
#endif
