/************************************************************************
 ** This file is part of the network simulator Shawn.                  **
 ** Copyright (C) 2004-2007 by the SwarmNet (www.swarmnet.de) project  **
 ** Shawn is free software; you can redistribute it and/or modify it   **
 ** under the terms of the BSD License. Refer to the shawn-licence.txt **
 ** file in the root of the Shawn source tree for further details.     **
 ************************************************************************/
#include "_legacyapps_enable_cmake.h"
#ifdef ENABLE_WS08

#include "legacyapps/ws08/ws08_delayed_transmission_model.h"
#include "sys/simulation/simulation_environment.h"
#include "sys/simulation/simulation_controller.h"
#include "sys/edge_model.h"
#include "sys/world.h"
#include "sys/misc/random/basic_random.h"
#include <math.h>

using namespace shawn;

namespace ws08
{

   // ----------------------------------------------------------------------
   Ws08DelayedTransmissionModel::
   Ws08DelayedTransmissionModel()
   {
   }

   // ----------------------------------------------------------------------
   Ws08DelayedTransmissionModel::
   ~Ws08DelayedTransmissionModel()
   {
   }

   // ----------------------------------------------------------------------
   bool 
   Ws08DelayedTransmissionModel::
   supports_mobility(void) 
      const throw(std::logic_error)
   {
      return world().edge_model().supports_mobility();
   }

   // ----------------------------------------------------------------------
   void 
   Ws08DelayedTransmissionModel::
   init(void) 
      throw()
   {
      TransmissionModel::init(); // assert( world() != NULL )
      const SimulationEnvironment& se =
      world().simulation_controller().environment();
      prob_ = se.required_double_param("delay_prob");

      // read out the additional optional parameter "max_delay"
      max_delay_ = se.optional_int_param("max_delay", 1);

      INFO(world().logger(), "ws08_delayed: Initialised. Delay probability is "
            << prob_ << std::endl << "Messages may be delayed for" << max_delay_
            << "rounds" << std::endl);
   }

   // ----------------------------------------------------------------------
   void 
   Ws08DelayedTransmissionModel::
   reset(void) 
      throw()
   {
      TransmissionModel::reset();

      while (!aired_messages_.empty())
      {
         MessageInfo* mi = aired_messages_.front();
         aired_messages_.pop();
         delete mi;
      }

      while (!delayed_messages_.empty())
      {
         MessageInfo* mi = delayed_messages_.front().second;
         delayed_messages_.pop_front();
         delete mi;
      }
   }

   // ----------------------------------------------------------------------
   void 
   Ws08DelayedTransmissionModel::
   send_message(TransmissionModel::MessageInfo& mi) 
      throw()
   {
      if (uniform_random_0e_1i() <= prob_)

      if (max_delay_> 1)
      {
         //sets the delay counter to a vale between 1 and max_delay_
         int delay_cnt_ = floor(uniform_random_0i_1e() * max_delay_) + 1;
         delayed_messages_.push_back(std::make_pair(delay_cnt_, &mi));
      } else
      delayed_messages_.push_back(std::make_pair(1, &mi));
      else
      queued_messages_.push(&mi);
   }

   // ----------------------------------------------------------------------
   void 
   Ws08DelayedTransmissionModel::
   deliver_messages() 
      throw()
   {
      // copy the queued messages to the aired_messages and clear the queue.
      while (!queued_messages_.empty())
      {
         aired_messages_.push(queued_messages_.front());
         queued_messages_.pop();
      }

      // Deliver messages. It is possible, that a processor receives a
      // message and resends it immediately. Therefore there are the two
      // queues queued_ and aired_messages. The following loop processes
      // the aired_messages_, whereby messages which are sent during the
      // loop, are packed into queued_messages_.
      while (!aired_messages_.empty())
      {
         MessageInfo* mi = aired_messages_.front();
         aired_messages_.pop();
         deliver_one_message(*mi);
      }

      if (max_delay_ == 1)
      // Copy the delayed messages into the aired_message_ queue for delivery
      // in the next round.

      {
         while (!delayed_messages_.empty())
         {
            aired_messages_.push(delayed_messages_.front().second);
            delayed_messages_.pop_front();
         }
      } else
      // read the delay counter of the delayed messages and either copy the message to
      // the aired_messages_ for delivery or reduce the delay counter by one

      {
         std::list<std::pair<int, TransmissionModel::MessageInfo*> >::iterator
         it;

         for (it = delayed_messages_.begin(); it != delayed_messages_.end(); it++)
         {
            if ((*it).first == 1)
            //messages should be delivered next round

            {
               aired_messages_.push((*it).second);
               delayed_messages_.erase(it);
               it--;
            } else
            //diminish the delay_counter

            {
               ((*it).first)--;
            }

         }
      }
   }

   // ----------------------------------------------------------------------
   void 
   Ws08DelayedTransmissionModel::
   deliver_one_message(TransmissionModel::MessageInfo& mi) 
      throw()
   {

      if (mi.msg_->is_unicast())
      {
         ABORT_INCONSISTENT_CONFIGURATION(
               "Unicast is not supported by the delayed transmission model. ");
      }

      for (EdgeModel::adjacency_iterator it = world_w().begin_adjacent_nodes_w(
                  *mi.src_), endit = world_w().end_adjacent_nodes_w(*mi.src_); it
            != endit; ++it)
      it->receive(ConstMessageHandle(mi.msg_));

      const Message* m = mi.msg_.get();
      if (m->has_sender_proc())
      (m->sender_proc_w()).process_sent_indication(ConstMessageHandle(
                  mi.msg_), shawn::Processor::SHAWN_TX_STATE_SUCCESS, 1);

      delete &mi;
   }

}

#endif
