/************************************************************************
 ** This file is part of the network simulator Shawn.                  **
 ** Copyright (C) 2004-2007 by the SwarmNet (www.swarmnet.de) project  **
 ** Shawn is free software; you can redistribute it and/or modify it   **
 ** under the terms of the BSD License. Refer to the shawn-licence.txt **
 ** file in the root of the Shawn source tree for further details.     **
 ************************************************************************/
#include "_legacyapps_enable_cmake.h"
#ifdef ENABLE_WS08

#include "legacyapps/ws08/ws08_processor.h"
#include "legacyapps/ws08/ws08_message.h"
#include "sys/simulation/simulation_controller.h"
#include "sys/node.h"
#include <iostream>
#include <limits>
#include <cmath>

namespace ws08
{
   const int Ws08Processor::UNKNOWN_DIST = std::numeric_limits<int>::max();
   // ----------------------------------------------------------------------
   Ws08Processor::
   Ws08Processor()
      : connected_      ( false ),
        gateway_        ( false ),
        hop_dist_       ( UNKNOWN_DIST ),
        send_period_    ( 6 ),
        predecessor_    ( NULL )
   {}
   // ----------------------------------------------------------------------
   Ws08Processor::
   ~Ws08Processor()
   {}
   // ----------------------------------------------------------------------
   void
   Ws08Processor::
   special_boot( void )
      throw()
   {
      gateway_ = true;
      connected_ = true;
      hop_dist_ = 0;
      
      send( new Ws08FloodingMessage( 1 ) );
   }
   // ----------------------------------------------------------------------
   void
   Ws08Processor::
   boot( void )
      throw()
   {
      const shawn::SimulationEnvironment& se = owner().world().
                                    simulation_controller().environment();
      if ( gateway_ )
      {
         send_period_ = se.optional_int_param( "flood_period", 
                                                send_period_ );
      }
      else 
      {
         send_period_ = se.optional_int_param( "state_period", 
                                                send_period_ );
      }
      
      int round = owner().world().simulation_round();
      event_handle_ = owner_w().world_w().scheduler_w().new_event( 
                                  *this, round + send_period_, NULL );
   }
   // ----------------------------------------------------------------------
   bool
   Ws08Processor::
   process_message( const shawn::ConstMessageHandle& mh ) 
      throw()
   {  
      const Ws08FloodingMessage* floodmsg = dynamic_cast<const Ws08FloodingMessage*> ( mh.get() );
      if ( floodmsg != NULL && owner() != floodmsg->source() )
		{
         handle_flooding_message_node( *floodmsg );
			return true;
		}

      const Ws08StateMessage* statemsg = dynamic_cast<const Ws08StateMessage*> ( mh.get() );
      if ( statemsg != NULL && owner() == statemsg->destination() )
      {
         if ( gateway_ ) 
            handle_state_message_gate( *statemsg );
         else 
            handle_state_message_node( *statemsg );
         
         return true;
      }
      
      return shawn::Processor::process_message( mh );
   }
   // ----------------------------------------------------------------------
   void
   Ws08Processor::
   work( void )
      throw()
   {
      if ( gateway_ )
      {
         int min = std::numeric_limits<int>::max(); 
         int max = std::numeric_limits<int>::min();
         double avg = 0;
         
         for ( std::map<const shawn::Node*, int>::iterator
                  it = network_nodes_.begin();
                  it != network_nodes_.end();
                  ++it )
         {
            if ( it->second < min ) min = it->second;
            if ( it->second > max ) max = it->second;
            avg += it->second;
         }
         avg /= (double)network_nodes_.size();
         
         INFO( logger(), "Number of nodes known to gate:  " << network_nodes_.size() );
         INFO( logger(), "  Min hops:  " << min );
         INFO( logger(), "  Max hops:  " << max );
         INFO( logger(), "  Avg hops:  " << avg );
      }
   }
   // ----------------------------------------------------------------------
   void 
   Ws08Processor::
   timeout( shawn::EventScheduler&, shawn::EventScheduler::EventHandle, 
            double, shawn::EventScheduler::EventTagHandle& ) 
      throw()
   {
      if ( gateway_ )
      {
         send( new Ws08FloodingMessage( 1 ) );
      }
      else if ( !gateway_ && predecessor_ )
      {
         send( new Ws08StateMessage( owner(), 
												 *predecessor_, 
												 hop_dist_ ) );
      }
      
      int round = owner().world().simulation_round();
      event_handle_ = owner_w().world_w().scheduler_w().new_event( 
                                     *this, round + send_period_, NULL );
   }
   // ----------------------------------------------------------------------
   void 
   Ws08Processor::
   handle_flooding_message_node( const Ws08FloodingMessage& flooding ) 
      throw()
   {      
      if ( hop_dist_ <= flooding.hops() )
         return;
      
      hop_dist_ = flooding.hops();
      predecessor_ = &flooding.source();
            
      // forward the flooding message 
      send( new Ws08FloodingMessage( hop_dist_ + 1 ) );
                  
      // if not connected, set to connected and start new timer for sending
      // state messages
      if ( !connected_ )
      {
         connected_ = true;
         int round = owner().world().simulation_round();
         event_handle_ = owner_w().world_w().scheduler_w().new_event( 
                                       *this, round + send_period_, NULL );
      }
   }
   // ----------------------------------------------------------------------
   void 
   Ws08Processor::
   handle_state_message_node( const Ws08StateMessage& state ) 
      throw()
   {
      // only connected nodes are allowed to forward messages (so predecessor
      // must not be 0, of course).
      if ( predecessor_ )
      {
         send( new Ws08StateMessage( state.initiator(),
                                     *predecessor_, 
                                     state.initiator_hops() ) );
      }
   }
   // ----------------------------------------------------------------------
   void
   Ws08Processor::
   handle_state_message_gate( const Ws08StateMessage& state ) 
      throw()
   {
      network_nodes_[ &state.initiator() ] = state.initiator_hops();
   }
   
}
#endif
