/***************************************************************************
 ** This file is part of the generic algorithm library Wiselib.           **
 ** Copyright (C) 2008,2009 by the Wisebed (www.wisebed.eu) project.      **
 **                                                                       **
 ** The Wiselib is free software: you can redistribute it and/or modify   **
 ** it under the terms of the GNU Lesser General Public License as        **
 ** published by the Free Software Foundation, either version 3 of the    **
 ** License, or (at your option) any later version.                       **
 **                                                                       **
 ** The Wiselib is distributed in the hope that it will be useful,        **
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of        **
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         **
 ** GNU Lesser General Public License for more details.                   **
 **                                                                       **
 ** You should have received a copy of the GNU Lesser General Public      **
 ** License along with the Wiselib.                                       **
 ** If not, see <http://www.gnu.org/licenses/>.                           **
 ***************************************************************************/
#ifndef CONNECTOR_REMOTE_UART_MESSAGE_H
#define CONNECTOR_REMOTE_UART_MESSAGE_H

#include "util/serialization/simple_types.h"

namespace wiselib
{

   template<typename OsModel_P,
            typename Radio_P = typename OsModel_P::Radio>
   class RemoteUartInMessage
   {
   public:
      typedef OsModel_P OsModel;
      typedef Radio_P Radio;
      typedef typename Radio::block_data_t block_data_t;
      typedef typename Radio::size_t size_t;
      typedef typename Radio::node_id_t node_id_t;
      typedef typename Radio::message_id_t message_id_t;
      // --------------------------------------------------------------------
      inline message_id_t command_type()
      { return read<OsModel, block_data_t, message_id_t>( buffer ); }
      // --------------------------------------------------------------------
      inline void set_command_type( message_id_t type )
      { write<OsModel, block_data_t, message_id_t>( buffer, type ); }
      // --------------------------------------------------------------------
      inline size_t payload_length()
      { return read<OsModel, block_data_t, size_t>( buffer + PAYLOAD_SIZE_POS ); }
      // --------------------------------------------------------------------
      inline uint64_t destination()
      { return read<OsModel, block_data_t, uint64_t>( buffer + DESTINATION_POS ); }
      // --------------------------------------------------------------------
      inline void set_destination( uint64_t dst )
      { write<OsModel, block_data_t, uint64_t>( buffer + DESTINATION_POS, dst ); }
      // --------------------------------------------------------------------
      inline uint64_t source()
      { return read<OsModel, block_data_t, uint64_t>( buffer + SOURCE_POS ); }
      // --------------------------------------------------------------------
      inline void set_source( uint64_t src )
      { write<OsModel, block_data_t, uint64_t>( buffer + SOURCE_POS, src ); }
      // --------------------------------------------------------------------
      inline block_data_t* payload()
      { return buffer + PAYLOAD_POS; }
      // --------------------------------------------------------------------
      inline void set_payload( size_t len, block_data_t *buf )
      {
         set_payload_length( len );
         memcpy( buffer + PAYLOAD_POS, buf, len);
      }
      // --------------------------------------------------------------------
      inline size_t buffer_size()
      { return 19 + payload_length(); }

   private:

      inline void set_payload_length( size_t len )
      { write<OsModel, block_data_t, size_t>( buffer + PAYLOAD_SIZE_POS, len ); }
      // --------------------------------------------------------------------
      /** Some remarks to the below positions:
       *    * PAYLOAD_POS = SOURCE_POS + 8 + 1: + 1 for "historical reasons"
       *      since our script, which understands remote uart messages,
       *      is written for an additional byte after SOURCE_POS - here, it
       *      is not needed and thus skipped
       *    * Node IDs are *here* uint64_t - however, that is also due to
       *      an application in another context. If only working on nodes,
       *      it can/should be exchanged with node_it_t (and sizeof(node_it_t)
       *      in the enum below)
       */
      enum data_in_positions
      {
         COMMAND_TYPE     = 0,
         PAYLOAD_SIZE_POS = sizeof(message_id_t),
         DESTINATION_POS  = PAYLOAD_SIZE_POS + 1,
         SOURCE_POS       = DESTINATION_POS + 8,
         PAYLOAD_POS      = SOURCE_POS + 8 + 1
      };
      // --------------------------------------------------------------------
      block_data_t buffer[Radio_P::MAX_MESSAGE_LENGTH];
   };

}

#endif
